/// \file standalone.c -- VS1063 Stand-Alone MP3 Player Source Code
/// Copyright (C) 2011 VLSI Solution Oy Tampere Finland

/*
  Note:

  DREQ is changed in normal firmware both in SDI interrupts when the
  stream buffer becomes too full, and whenever the stream buffer becomes
  empty enough.

  So, we can use DREQ ourselves as we wish and access to it does not need to
  be protected by interrupt Disable()/Enable(). However, the normal decoding
  will still keep raising DREQ when it sees that there is space for new data.

- VS1063:
  + I2S
  + higher reference
- IN VS1063:
  + SCLK = GPIDATA8
  + XCS  = GPIDATA9
  + SI   = GPIDATA10
  + XDCS = GPIDATA11


- Exiting recoding now from any button. Key scan is not well performed
  during encoding, because applAddr is not called and we have no time base.
  We should use a timer to get 16Hz trigger for key scan.
- Now ends recording nicely and creates 100% valid files.
- If format is WAV, fix the WAV header.
- Finding free space takes a few seconds, so we now cache the free space.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "standalone.h"
#include "vs1063.h"

//#define USE_DEBUG

#undef puts
#undef puthex
#undef perror

/* See standalone.h for feature #defines. */

#if defined(SAVE_VOLUME) || defined(SAVE_POSITION)
  #define SAVE_STUFF
#endif

extern volatile s_int16 *srcWp, *srcRp;

extern __y u_int16 g_dcthi[2048];
auto void play_frame(void);
//void putch(register __a0 int ch);
void putword(register __a0 int ch);

#if 0 //defined(USE_DEBUG) && !defined(UART_BUFFERED)
  #define ENABLE_SCI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_DAC))
  #define ENABLE_SCI_SDI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_SDI)|(1<<INT_EN_DAC))
#else
  #define ENABLE_SCI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_DAC)|(1<<INT_EN_RX))
  #define ENABLE_SCI_SDI_DAC_RX ((1<<INT_EN_SCI)|(1<<INT_EN_SDI)|(1<<INT_EN_DAC)|(1<<INT_EN_RX))
#endif

#ifndef NOCHECKFILETYPE
  #define CHECKFILETYPE CheckFileType1063
#endif/*NOCHECKFILETYPE*/


s_int16 hcShift;

#ifdef RECORDER /* if standalone player + recorder, disable some features */
  //#undef HDATCHECK           /* no timeout               -14 words */
  //#undef CHECKFILETYPE       /* no filename suffix check -27 words */
#endif/*RECORDER*/


//#define CTRL3_UPDATE_VOL        (1<<15) /*not needed for vs1053*/
#define CTRL3_BY_NAME           (1<<8) /*locate by name*/
#define CTRL3_RECORD_ON         (1<<7)
#define CTRL3_AT_END            (1<<6)
#define CTRL3_NO_NUMFILES       (1<<5)
#define CTRL3_PAUSE_ON          (1<<4)
#define CTRL3_FILE_READY        (1<<3)

#define CTRL3_PLAY_MODE_MASK    (3<<1)
#define CTRL3_PAUSE_AFTER_PLAY  (3<<1)
#define CTRL3_PAUSE_BEFORE_PLAY (2<<1)
#define CTRL3_LOOP_SONG         (1<<1)
#define CTRL3_PLAY_NEXT         (0<<1)

#define CTRL3_RANDOM_PLAY       (1<<0) /* SCI_AICTRL3 */

#ifndef GPIO_IDATA
  #define GPIO_IDATA GPIO_DATA
#endif


#define SPI_xCS   1 /* GPIO */
#define SPI_CLK   8
#define SPI_MISO  4
#define SPI_MISO_SHIFT -3
#define SPI_xCS2  16
#define SPI_MOSI  1 /* DREQ */



#ifdef USE_DEBUG
static char hex[] = "0123456789ABCDEF";
void puthex(u_int16 d) {
    register i;
    for (i=0;i<4;i++) {
	putchar(hex[(d>>12)&15]);
	d <<= 4;
    }
}
void puthex2(u_int16 d) {
    register i;
    putchar(hex[(d>>4)&15]);
    putchar(hex[(d>>0)&15]);
}
#endif


typedef enum {
    spi_pause=0,
    spi_file=1,
    spi_seek=2,
    spi_waitdata=3,
    spi_data=4,
    spi_zeros=5,
    spi_record=6,
    spi_waitima=7
} SPISTATE;

SPISTATE spiState, oldSpiState;
u_int16 spiCnt;


struct SPIX { /* do not touch! ASM uses the structure */
    s_int32 size;
    u_int32 sector;
    u_int16 fragSize; /* TODO: make this 32-bit and multiply fragment list clusters by sectorsPerCluster. */
    __y struct FRAGMENT *fragment;
} spix;


#if 1
auto u_int16 MmcCommand(register __b0 s_int16 cmd, register __d u_int32 arg);
#else
auto u_int16 MmcCommand(register __b0 s_int16 cmd, register __d u_int32 arg) {
    register __b1 u_int16 t;
    #if 1
        puts("Cmd(");
	puthex(cmd);
	puthex(arg>>16);
	puthex(arg);
	puts(")");
    #endif
    /* some MMC brands require the extra clock pulses     */
    SpiSendClocks();
    SpiSendClocks();
    USEX(GPIO_ODATA) &= ~SPI_xCS2;
  
    while (SpiSendReceiveMmc(0xff00,8) != 0xff) {
	/* Unexpected busy signal from MMC */
        #ifdef USE_DEBUG
	    putchar('u');
            putchar('\n');
        #endif
    }
    SpiSendReceiveMmc(cmd, 16); /* top half normally 0xff */
    SpiSendReceiveMmc(arg>>16, 16);
    SpiSendReceiveMmc(arg, 16);
    t = SpiSendReceiveMmc(0x9500, 8);    /* Valid CRC for init, then don't care */
  
    cmd = 100;               /* timeout value, try max. 100 times */
    while (cmd && (t & 0x80)) { /* R1 response not detected */
	t = SpiSendReceiveMmc(0xff00, 8);
	cmd--;               /* decrease the timeout value */
    }
    #ifdef USE_DEBUG
        puthex(t);
        putchar('\n');
    #endif
    return t;
}
#endif

#if 1
//auto void ReadDiskSectorTo(register __a u_int32 sector);
auto void ReadDiskSectorTo(register __a u_int32 sector,
			   register __i2 u_int16 *buffer);
#else
auto void ReadDiskSectorTo(register __a u_int32 sector,
			   register __i2 u_int16 *buffer) {
    register s_int16 i;
  
    #if 0 && defined(USE_DEBUG)
        puthex(sector>>16);
        puthex(sector);
        puts(" SECTOR");
    #endif

    MmcCommand(MMC_READ_SINGLE_BLOCK|0x40,(sector&0x7fffffffUL)<<hcShift);
    do {
        i = SpiSendReceiveMmc(0xff00,8);
        #if 0 && defined(USE_DEBUG)
          puthex(i);
          puts(" wait");
        #endif
    } while (i == 0xff);

    #if 0
      if (i != 0xfe) {
          SpiSendClocks();/*USEX(GPIO_ODATA) |= SPI_xCS2;*/
          #ifdef USE_DEBUG
            puts("wfd fail");
          #endif
          return;
      }
    #endif
    
    for (i=0; i<512; i+=2) {
        *buffer++ = SpiSendReceiveMmc(0xffff,16);
    }
    SpiSendReceiveMmc(0xffff,16); /* discard crc */

    /* generate some extra SPI clock edges to finish up the command */
    SpiSendClocks();
    SpiSendClocks();
    return;
}
#endif

#if 0
struct Fat {
    byte BS_jmpBoot[3];         /**< 0 x86 Boot Jump Code */
    byte BS_OEMName[8];         /**< 3 Formatter's name, usually "MSWIN4.1" */
      
    word BPB_BytsPerSec;        /**< 11 Bytes per sector (512) */
    byte BPB_SecPerClus;        /**< 13 Sectors per Cluster (1,2,4,8,..,128) */
    word BPB_RsvdSecCnt;        /**< 14 Reserved sectors (1 (32 for FAT32)) */
    byte BPB_NumFATs;           /**< 16 Number of FATs (2) */
    word BPB_RootEntCnt;        /**< 17 FAT12/16 n of root dir entries */
    word BPB_TotSec16;          /**< 19 Old sector count (0 for FAT32) */
    byte BPB_Media;             /**< 21 Media Type (eg 0xF8) */
    word BPB_FATSz16;           /**< 22 Size of one FAT16 in sectors */
    word BPB_SecPerTrk;         /**< 24 Old CHS Sectors Per Track */
    word BPB_NumHeads;          /**< 26 Old CSH Number of Heads */
    u_32 BPB_HiddSec;           /**< 28 n of sectors before this volume */
    u_32 BPB_TotSec32;          /**< 32 New sector count (0 for FAT12/16) */
     
    /** FAT type specific extensions */
    union Extensions{   
  
        /** FAT12/16 specific extensions to Bios Parameter Block*/
        struct Fat16Specific {    
          byte BS_DrvNum;         /**< 36 DOS INT13 Drive Number (0x80=HD)*/
          byte BS_Reserved1;      /**< 37 For WINNT; Format to 0 */
          byte BS_BootSig;        /**< 38 0x29 if next 3 fields are present */
          byte BS_VolID[4];       /**< 39 Volume ID (usually format datetime) */
          byte BS_VolLab[11];     /**< 43 Volume Label */
          byte BS_FilSysType[8];  /**< 54 Decorative name of fs, eg "FAT16   "*/
        } _16;
  
        /** FAT32 specific extensions to Bios Parameter Block*/
        struct Fat32Specific {    
          u_32 BPB_FATSz32;       /**< 36 Size of one FAT32 in sectors */
          word BPB_ExtFlags;      /**< 40 Flags; active FAT number etc */
          word BPB_FSVer;         /**< 42 File System Version (0x0000) */
          u_32 BPB_RootClus;      /**< 44 Start cluster of Root Dir. (2) */
          word BPB_FSInfo;        /**< 48 Start sector of FSINFO in Resvd area */
          word BPB_BkBootSec;     /**< 50 Sector in Resvd area for BkBoot (6) */
          byte BPB_Reserved[12];  /**< 52 Reserved, Always 0. */
          byte BS_DrvNum;         /**< 64 DOS INT13 Drive Number (0x80=HD) */
          byte BS_Reserved1;      /**< 65 For WINNT, Format to 0 */
          byte BS_BootSig;        /**< 66 0x29 if next 3 fields are present */
          byte BS_VolID[4];       /**< 67 Volume ID (usually format datetime) */
          byte BS_VolLab[11];     /**< 71 Volume Label */
          byte BS_FilSysType[8];  /**< 82 Decorative name of fs, eg "FAT32   "*/
        } _32;
      } ext;
    } fat;
#endif


/* Get Little-endian values from a sector read to stream_buffer with
   arbitrary byte alignment. */
#if 1
auto u_int16 GetByte(register __c0 u_int16 n);
#else
auto u_int16 GetByte(register __c0 u_int16 n) {
    register u_int16 *p = stream_buffer[n>>1];
    if (n & 1) {
	n = *p;
    } else {
	n = *p >> 8;
    }
    return n & 0xff;
}
#endif
#if 1
auto u_int16 GetWord(register __c0 u_int16 n);
#else
auto u_int16 GetWord(register __c0 u_int16 n) {
    return GetByte(n) | (GetByte(n+1)<<8);
}
#endif
#if 1
auto u_int32 GetLong(register __c0 u_int16 n);
#else
auto u_int32 GetLong(register __c0 u_int16 n) {
    return GetWord(n) | ((u_int32)GetWord(n+2)<<16);
}
#endif

extern __y u_int16 fileName[6];
#ifdef FATINFO_IN_Y
__y struct FATINFO fatInfo;
#else
struct FATINFO fatInfo;
#endif

#if 0 //def USE_DEBUG
void PrintDiskSector(void) {
    register int i;
    register int j;
    for (i=0;i<512;i+=16) {
	puthex(i);
	putchar(' ');
	putchar(' ');
	for (j=0;j<16;j++) {
	    puthex2(GetByte(i+j));
	    putchar(' ');
	}
	putchar(' ');
	for (j=0;j<16;j++) {
	    register int t = GetByte(i+j);
	    if ((t & 0x7f) < 0x20)
		putchar('?');
	    else
		putchar(t);
	}
	putchar('\n');
    }
}
#endif

#if 1
auto u_int16 InitFileSystem(void);
#else
auto u_int16 InitFileSystem(void) {
    /* Load MBR -- the first sector on disk */
    ReadDiskSectorTo(0, stream_buffer);
  
    if (GetWord(510) != 0xaa55)
	return 0xaa55;

    #if 0 && defined(USE_DEBUG)
        /* Ok, it should be a MBR sector. Let's verify */
        if (stream_buffer[255] != 0x55aa) {
        #ifdef USE_DEBUG
              puts("!MBR");
        #endif
            return -1; /* sector 0 is not MBR. */
        }
    #endif

#if 0 && defined(USE_DEBUG)
    /* This checks that partition 1 is active. Alter code to allow
       other partition configurations. */
    if (GetByte(0x1be) == 0x80) {
    } else {
	puthex(GetByte(0x1be));
	puts(" PF!");
    }
#endif

    /* Now leave MBR and load sector 0 of partition */
    /* check for boot sector instead of partition table */
    if (GetLong(0x36)!= (((long)'1'<<24)|((long)'T'<<16)|((long)'A'<<8)|'F') &&
        GetLong(0x52)!= (((long)'3'<<24)|((long)'T'<<16)|((long)'A'<<8)|'F')
        ) {
        #ifdef USE_DEBUG
          PrintDiskSector();
          puts("");
        #endif
        fatInfo.currentSector = GetLong(0x1c6);
        ReadDiskSectorTo(fatInfo.currentSector, stream_buffer);
    }
    #ifdef USE_DEBUG
      PrintDiskSector();
    #endif
    
    {
        u_int16 BPB_BlksPerSec;

        /* Determine FAT Type (16/32) */
        /* This should be done better, but it'll do for now. */
        fatInfo.FilSysType = GetWord(54+3); /* 0x3231 for FAT12 */
        fatInfo.BPB_RootEntCnt = GetWord(17);
        /* FAT32 does not have separate root entries. */
        fatInfo.IS_FAT_32 = fatInfo.BPB_RootEntCnt ? 0 : 1;
        #ifdef USE_DEBUG
          puthex(fatInfo.FilSysType);
          puts("fsys");
        #endif
        
        #if 0 && defined(USE_DEBUG)
          if (fatInfo.IS_FAT_32) {
              puts("FAT32");
          } else {
              puthex(fatInfo.BPB_RootEntCnt);
              puts(" FAT");
              puthex(GetWord(19));
              puts(" TSc16");
          }
          puthex(GetWord(54+3)); /* 3231 == '12' 3631= '16' 0000=FAT32 */
          puts(" fst");
        #endif

        /* OK, let's calculate */
        /* First, let's get rid of the idea that we have byte addresses
           in the file system. Nope, let's only deal in physical disk
           sectors of 512 bytes. First we convert the FAT byter per sector
           value to "512B disk sectors per fat sector" value. */

        BPB_BlksPerSec = GetWord(11) >> 9;

        /* Then we adjust the Sector per Cluster to mean physical disk
           sectors. in 99% of the cases it is already so because bytes
           per sector almost always is 512 in FAT. Maximum cluster size
           is 65536 bytes (128 disk sectors). */

        fatInfo.fatSectorsPerCluster = GetByte(13)/*BPB_SecPerClus*/ * BPB_BlksPerSec;
        #if 0 && defined(USE_DEBUG)
          puthex(fatInfo.fatSectorsPerCluster);
          puts(" FatSPC");
        #endif
	fatInfo.totSize = GetWord(19);
	if (fatInfo.totSize == 0) {
	    fatInfo.totSize = GetLong(32); /*TotSec32*/
	}
        fatInfo.fatStart = fatInfo.currentSector +
            (u_int32)GetWord(14)/*BPB_RsvdSecCnt*/ * BPB_BlksPerSec;
            #if 0 && defined(USE_DEBUG)
          puthex(fatInfo.fatStart);
          puts(" fatS");
            #endif
	fatInfo.rootStart = GetWord(22);/*BPB_FATSz16*/
        if (fatInfo.rootStart == 0) {
            if (!fatInfo.IS_FAT_32) 
                return 0x0b; /* should be FAT32; can not find root directory */
            fatInfo.rootStart = GetLong(36);/*BPB_FATSz32*/
        }
        /* BPB_NumFATs * BPB_BlksPerSec fits in a word (8+7 bits) */
        fatInfo.rootStart =
	    fatInfo.rootStart * (GetByte(16)/*BPB_NumFATs*/ * BPB_BlksPerSec)
            + fatInfo.fatStart;
            #if 0 && defined(USE_DEBUG)
          puthex(fatInfo.rootStart);
          puts(" rootS");
            #endif
	fatInfo.dataStart = fatInfo.rootStart + (fatInfo.BPB_RootEntCnt >> 4)
	    - (fatInfo.fatSectorsPerCluster << 1); /*first cluster is 2*/
	/* Because dataStart has been moved, remove 2 from the result. */
	fatInfo.numClusters = (fatInfo.totSize - fatInfo.dataStart) / fatInfo.fatSectorsPerCluster - 2;

	/* The checks are not yet in the ASM version. */
	if (fatInfo.numClusters < 4085) {
	    fatInfo.FilSysType = 0x3231; /* FAT12 */
	} else if (fatInfo.numClusters < 65525) {
	    /*FAT16: check that we agree*/
	    if (fatInfo.IS_FAT_32)
		return 0xbad0;
	} else {
	    /*FAT32: check that we agree*/
	    if (!fatInfo.IS_FAT_32)
		return 0xbad1;
	}

            #if 0 && defined(USE_DEBUG)
          puthex(fatInfo.dataStart);
          puts(" dataS");
  //      puthex(fatInfo.clusters);
  //      puts(" Clustrs");
            #endif
    }
    return 0;
}
#endif

#if 0
/** FAT/VFAT directory record union */
union DirRecordUnion {
    /** Standard directory entry */
    struct Entry {
      byte Name[11]; 0
      byte Attr;     11
      byte NTRes;    12
      byte CrtTimeTenth; 13
      word CrtTime;  14
      word CrtDate;  16
      word LstAccDate; 18
      word FstClusHi; 20
      word WrtTime;   22
      word WrtDate;   24
      word FstClusLo; 26
      u_32 FileSize;  28
    } entry;
  
    /** Extended directory entry */
    struct LongEntry {
      byte Ord;      0
      word Name1[5]; 1 /**< characters 1-5 */
      byte Attr;     11
      byte Type;     12 /**< entry type, zero=long name component */
      byte Chksum;   13
      word Name2[6]; 14 /**< characters 6-11 */
      word FstClusLO; 26 /**< zero for long entry */
      word Name3[2]; 28 /**< characters 12-13 */
    } longentry;
  };
#endif


//#define MAX_FRAGMENTS 28//16//22
__y struct FRAGMENT fragments[MAX_FRAGMENTS];

#ifndef FAT12_SUPPORT /* not yet in asm */
auto __y struct FRAGMENT *FragmentList(register __i2 __y struct FRAGMENT *frag,
				       register __b u_int32 fatCluster);
#else
auto __y struct FRAGMENT *FragmentList(register __i2 __y struct FRAGMENT *frag,
				       register __b u_int32 fatCluster) {
    register __d0 u_int16 offset;
  
    /* fatCluster means now current cluster, starting from 0, not 2. */
    /* note that startcluster=2 is already compensated in fatInitGlobals */
    #if 0 && defined(USE_DEBUG)
        puthex(fatCluster>>16);
        puthex(fatCluster);
        puts(" fatC");
    #endif
    #if 0
        frag->size = 0;
        frag->start = fatCluster * fatInfo.fatSectorsPerCluster + fatInfo.dataStart;
    #endif

    fatInfo.currentSector = -1;
    goto l55; /* read first sector */
  
    while (1) {
	u_int32 t;

#ifdef FAT12_SUPPORT
	/* TODO: use cluster count to set FilSysType! */
	if (fatInfo.FilSysType == 0x3231) {
	    /*
	      00 01 11 22 23 33 44 45 55 ..
	      yz z..
	     */
	    u_int16 fof = fatCluster + fatCluster / 2;
	    u_int16 sec = fof / 512;
	    u_int16 off = fof % 512;

	    if (fatInfo.currentSector != sec + fatInfo.fatStart) {
		fatInfo.currentSector = sec + fatInfo.fatStart;
		ReadDiskSectorTo(sec + fatInfo.fatStart, stream_buffer);
	    }
	    if (off >= 510) {
		ReadDiskSectorTo(sec + 1 + fatInfo.fatStart, stream_buffer+512/2);
	    }
	    t = GetWord(off);
	    if (fatCluster & 1) {
		t >>= 4;
	    }
	    t &= 0x0fff;
            if (t >= 0x0ff8U)
                t = 0x0fffffffUL;
#if 0
	    puthex(t);
	    putchar('.');
	    putchar(' ');
#endif
	} else
#endif/*FAT12_SUPPORT*/
	if (fatInfo.IS_FAT_32) {
	    t = GetLong(offset) & 0x0fffffffUL;
	} else {
            #if 0 && defined(USE_DEBUG)
	    {
		register int i;
		for (i=0;i<256;i++) {
		    t = GetByte(i);
		    puthex(t);
		    putchar(' ');
		    if ((i&7)==7)
			putchar('\n');
		}
	    }
            #endif

            t = GetWord(offset);
            if (t >= 0xfff8U)
                t = 0x0fffffffUL;
        }


        #if 0 && defined(USE_DEBUG)
          puthex(t>>16);
          puthex(t);
          puts(" c");
        #endif
        frag->size += fatInfo.fatSectorsPerCluster;
        ++fatCluster;
        if (frag->size >= 32768U || t != fatCluster) {
            /* if not sequential cluster, start next fragment */
            #if 0 && defined(USE_DEBUG)
              puts(" frag");
            #endif
            fatCluster = t;
            if (frag == &fragments[MAX_FRAGMENTS-1] ||
                t >= 0x0ffffff8UL) {
                frag->start |= LAST_FRAGMENT;
                return frag+1;
            }
            frag++;
        l55:
            frag->size = 0;
            frag->start =
                fatCluster * fatInfo.fatSectorsPerCluster + fatInfo.dataStart;

#ifndef FAT12_SUPPORT
	    /* Hack for FAT12 */
	    if (fatInfo.FilSysType == 0x3231) {
		frag->size = 0xffffUL; /* FAT12 */
		return frag+1;
	    }
#endif/*FAT12_SUPPORT*/
        } else {
            offset += (fatInfo.IS_FAT_32 ? 4 : 2);
            offset &= 511;
            if (offset != 0)
                continue;
        }
read:
#ifdef FAT12_SUPPORT
	if (fatInfo.FilSysType == 0x3231) {
	} else
#endif/*FAT12_SUPPORT*/
        /* If FAT page is not already in memory, load it. */
        if (fatInfo.IS_FAT_32) {
            offset = ((u_int16)fatCluster & 0x7F) << 2;
            ReadDiskSectorTo((fatCluster >> 7) + fatInfo.fatStart,
			     stream_buffer);
        } else {
            offset = ((u_int16)fatCluster & 0xff) << 1;
            ReadDiskSectorTo((fatCluster >> 8) + fatInfo.fatStart,
			     stream_buffer);
        }
    }
}
#endif

u_int16 gFileNum[2];
#ifdef RECORDER
u_int16 playbackVol; /*temporary store for volume during encoding mode*/
struct REC {
    u_int16 record;
    u_int16 rd;
    u_int16 wr;
    s_int16 fill;
} record;
//free_y is 0xfa14..0xffff
#define YBUF ((__y s_int16 *)0xe000)
#define YBUFSZ 0x1800 //n*256
#endif/*RECORDER*/


#ifdef __VSDSP__
auto s_int16 OpenFile(register __c0 u_int16 fileNum);
#else
auto s_int16 OpenFile(register __c0 u_int16 fileNum) {
    register __i1 __y struct FRAGMENT *curFragment = &fragments[0];
    register __i0 __y struct FRAGMENT * __y nextFragment;
    
    /* Start at the start of root directory. */
    
    if (fatInfo.IS_FAT_32) {
	nextFragment = FragmentList(curFragment, 2);
    } else {
	curFragment->start = fatInfo.rootStart | LAST_FRAGMENT;
	curFragment->size  = (fatInfo.BPB_RootEntCnt >> 4); /* /sectorsPerCluster?*/
	nextFragment = curFragment + 1;
    }
    gFileNum[0] = fileNum;
    gFileNum[1] = 0;
    return HandleDir(curFragment, nextFragment);
}
#endif

#ifdef CHECKFILETYPE
#if 0
#define MKID(a,b,c) ((a)|((b)<<8)|((u_int32)(c)<<16))
auto s_int16 CheckFileType(register __a u_int32 name) {
    name &= 0x00ffffffUL;
    if (name == MKID('W','A','V'))
	return -1;
    if (name == MKID('F','L','A'))
	return -1;
    if (name == MKID('M','P','3'))
	return -1;
    if (name == MKID('M','I','D'))
	return -1;
    if (name == MKID('W','M','A'))
	return -1;
    if (name == MKID('A','S','F'))
	return -1;
    if (name == MKID('A','A','C'))
	return -1;
    if (name == MKID('M','P','4'))
	return -1;
    if (name == MKID('M','4','A'))
	return -1;
    if (name == MKID('3','G','P'))
	return -1;
    if (name == MKID('3','G','2'))
	return -1;
    if (name == MKID('O','G','G'))
	return -1;
    return 0;
}
#endif/*1*/
#endif/*CHECKFILETYPE*/

/*147 words*/
auto s_int16 HandleDir(register __i2 __y struct FRAGMENT *curFragment,
		       __y struct FRAGMENT *nextFragment) {
    register __c1 int i = 0;
    register u_int32 currentSector = curFragment->start /*& 0x7fffffffUL */;
  
#pragma msg 36 off
    goto first;
  
    while (1) {
	register __d1 u_int16 fn = GetByte(i+0); /* first char of filename */
  
#pragma msg 36 on
	/* We are now looking at FAT directory structure. */
	/* Is current file a regular file? */
	if (fn == 0)
	    goto eod; /*end of directory*/
	if (fn != 0xe5) { /* Not deleted */
            #ifdef WITH_SUBDIRECTORIES
	    /* Is it a subdirectory? */
	    if ((GetByte(i+11/*Attr*/) & 0x10) && fn != '.') {
		/* TODO: do not call if FAT12! */
#ifndef FAT12_SUPPORT
		if (fatInfo.FilSysType != 0x3231)
#endif
		{
		    if (HandleDir(nextFragment,
				  FragmentList(nextFragment,
					       ((u_int32)GetWord(i+20)<<16)
					       +GetWord(i+26))) < 0)
			return -1;
		    memset(stream_buffer, 0, 512/2);
		    /*reread sector*/
		    ReadDiskSectorTo(currentSector, stream_buffer);
		}
	    } else /* it was a subdirectory */
            #endif
		/* Attributes: NO directory, NO volume id, NO system, NO hidden */
	    if ((GetByte(i+11/*Attr*/) & 0xde) == 0
            #ifdef CHECKFILETYPE
                    /*
                      check supported files from name suffix:
                      vs1053: wav mp3 mid wma asf mp4 aac m4a ogg fla
                    */
		&& CHECKFILETYPE(GetLong(i+8))
            #endif
		) {
  
		/* It is a regular file. */
                #if defined(SCI_UI)
		  /* If open-by-name, check name and open file if match. */
#pragma msg 240 off
		  if (USEX(SCI_AICTRL3) & CTRL3_BY_NAME) {
		    register int j, t = 0;
		    register u_int16 *p = stream_buffer + i/2;
		    register __y u_int16 *d = fileName;
		    p[5] &= 0xff00; /* clear the attribute byte */
		    for (j=0; j<6; j++) {
			t |= *d++ ^ *p++;
		    }
		    if (t == 0) {
			USEX(SCI_AICTRL0) = gFileNum[1];
			goto thisfile;
		    }
		  }
#pragma msg 240 on
                #endif

		/* remember the highest used cluster */
		{
		    u_int32 cluster =
			((u_int32)GetWord(i+20) << 16) + GetWord(i+26);
#ifdef RECORDER
		    if (mmc.usedTopCluster < cluster) {
			mmc.usedTopCluster = cluster;
#if 0
			puthex(mmc.usedTopCluster>>16);
			puthex(mmc.usedTopCluster);
			puts(" =c");
#endif
		    }
#endif
		}

                gFileNum[1]++; /* total file count */
                if (gFileNum[0]-- == 0) {
                thisfile:
                  #ifdef SCI_UI
		    /* Put the 8.3-character filename to memory so the user
		       can read it. Two bytes per word, big-endian. */
		    {   /* 13 words */
			register int j;
			register u_int16 *p = stream_buffer + i/2;
			register __y u_int16 *d = fileName;
			for (j=0;j<6;j++) {
			    *d++ = *p++;
			}
		    }
                  #endif
                    /* ------------ FILE FOUND ------------- */
                    fatInfo.fileSize = GetLong(i+28); /*do first!*/
                    FragmentList(&fragments[0],
                                 ((u_int32)GetWord(i+20)<<16) + GetWord(i+26));
                    return -1; /* File found, All OK return */
                }
            } /* normal file */
        } /* 0xe5 / 0x00 deleted or unused entry */
        i = (i + 32) & 511;
        if (i == 0) {
            currentSector++;
            if (--(curFragment->size) == 0) {
                if ((s_int16)(curFragment->start>>16) < 0) {
		eod:
                #if defined(SCI_UI)
		    USEX(SCI_AICTRL3) &= ~CTRL3_BY_NAME;
                #endif
                    return (s_int16)gFileNum[1];
                }
                curFragment++;
                currentSector = curFragment->start /*& 0x7fffffffUL*/;
            }
        first:
            #if 0 && defined(USE_DEBUG)
                puthex(currentSector);
                puts(" read");
            #endif
	    memset(stream_buffer, 0, 512/2);
            ReadDiskSectorTo(currentSector, stream_buffer);
        }
    }
}


#ifdef HDATCHECK
  u_int16 hdatCheck;
#endif
extern __y u_int16 hwSampleRate;

/* Can be done for all formats, because IdleHook is run as the main thread
   and not in interrupts. Whenever IdleHook is called, there either is
   not enough stream data anyway, or the decoder is outputting samples
   and there is not enough space in the audio buffer (thus not reading
   stream). */
u_int16 __x * __x stream_save;

#ifndef SCI_UI
__y struct {
    u_int16 Data; /* must be first entry! Used by c.s */
    u_int16 Cnt;
    u_int16 Strobe;
    u_int16 Timer; /* Data,Cnt,Strobe,Timer must be in this order (buts.s) */
    u_int16 Key;
    s_int16 KeyTime;
} ui;
#endif

#if 1
  #ifdef ENABLE_UI
    s_int16 ApplAddr(register __i0 s_int16 **d, register __a1 s_int16 mode,
                     register __a0 s_int16 n);
  #endif/*ENABLE_UI*/

  #ifdef SCI_UI
    s_int16 ApplAddrSci(register __i0 s_int16 **d, register __a1 s_int16 mode,
                        register __a0 s_int16 n);
  #endif/*SCI_UI*/
#else/*1*/
  #ifdef ENABLE_UI
    s_int16 ApplAddr(register __i0 s_int16 **d, register __a1 s_int16 mode,
                     register __a0 s_int16 n) {
        /*Don't check mode, all we get is APPL_AUDIO.*/
        //if (mode == APPL_AUDIO)
        {
            register u_int16 t = hwSampleRate/16;
            /* generate a timebase for User Interface */
            ui.Cnt += n;
            if (ui.Cnt > t) {
                ui.Cnt -= t;
                ui.Timer++;
                ui.Strobe = 1;
            }
        }
    }
  #endif/*ENABLE_UI*/
#endif


#if 1
auto u_int32 NextSector(void);
#else
auto u_int32 NextSector(void) {
    spix.sector++;
    if (spix.fragSize-1 == 0) {
	if ((s_int32)spix.sector < 0) /* was last fragment */
	    return -1;
	spix.fragment++;
	spix.sector  = spix.fragment->start /*& 0x7fffffffUL*/;
	spix.fragSize = spix.fragment->size; /* *fatInfo.sectorsPerCluster*/
    } else {
	spix.fragSize = spix.fragSize-1;
    }
    return 0;
}
#endif

#if 0 && defined(RECORDER)
/* Note: sector is not logical sector number, it is directly MMC argument */
auto void WritePhysicalSectorY(register __d u_int32 sector,
			       register __i3 __y u_int16 *dataBufPtr) {
    register u_int16 c;
    s_int16 to = 0;

    c = MmcCommand(MMC_WRITE_BLOCK|0x40, sector & 0x7fffffff);
    while (c != 0x00) {
	//wait for BUSY token, if you get 0x01(idle), it's an ERROR!
	c = SpiSendReceiveMmc(0xff00, 8);
//putch(c);
	to++;
	if (c == 0x01 || to < 0) {
	    return; //ERROR
	}
    }
    //MMC_XCS = MMC_SELECTED;
    SpiSendReceiveMmc(0xfffe, 16);
    for (c=0;c<256;c++) {
	SpiSendReceiveMmc(*dataBufPtr++, 16);
    }
    SpiSendReceiveMmc(0xffff, 16); /* send CRC */
    
    //c = SPIGetChar(); //This prints xxx00101, (usually e5) when data ok
    #ifdef USE_DEBUG
      c = SpiSendReceiveMmc(0xff00, 8);
      puthex2(c);
      puts(" response");
    #endif
    #if 0
      while ( SpiSendReceiveMmc(0xff00, 8) != 0xff)
	  ;
      SpiSendReceiveMmc(0xffff, 16);
      SpiSendReceiveMmc(0xffff, 16);
      SpiSendClocks();
      SpiSendClocks();
    #endif
} 
#endif/*0 && RECORDER*/

#if 1
auto u_int16 swap(register __a0 u_int16 val);
#else
auto u_int16 swap(register __a0 u_int16 val) {
    register u_int32 v = val<<8;
    return (u_int16)(v>>16)|(u_int16)v;
}
#endif


void SaveWord(register __c0 u_int16 addr, register __c1 u_int16 value);
#ifdef SAVE_VOLUME
int volChanged = 0;
#endif

#ifdef SAVE_STUFF
  #define SPI_WREN  0x06
  #define SPI_WRDI  0x04
  #define SPI_RDSR  0x05
  #define SPI_WRSR  0x01
  #define SPI_READ  0x03
  #define SPI_WRITE 0x02
  
  register __a1 short SpiDelay(void);
  auto u_int16 SpiSendReceiveSpi(register __a0 u_int16 datTopAligned, register __a1 s_int16 bits);
  
  #if 1
  void SaveWord(register __c0 u_int16 addr, register __c1 u_int16 value);
  #else
  void SaveWord(register __c0 u_int16 addr, register __c1 u_int16 value) {
      USEX(GPIO_ODATA) &= ~(SPI_xCS2|SPI_xCS);
      SpiSendReceiveSpi(SPI_WREN<<8, 8);
      USEX(GPIO_ODATA) |= SPI_xCS;
      USEX(GPIO_ODATA) &= ~SPI_xCS;
      SpiSendReceiveSpi(SPI_WRITE<<8, 8);
      SpiSendReceiveSpi(addr/*volume=26*/, 16);
      SpiSendReceiveSpi(value, 16);
      USEX(GPIO_ODATA) |= SPI_xCS|SPI_xCS2;//SPI_CLK;
  }
  #endif
  //SaveWord(26/*volume=26*/, USEX(SCI_VOL));
#endif

#ifdef SHUFFLE_PLAY
u_int16 shuffle_offset = 0;
#endif


/*
  Called before HALT
 */
void UserHook(void) {
#if 1
    /* Sometimes when audio buffer is empty when samplerate gets changed,
       SetRate() in vs1063a overwrites interrupt's change of the high 4
       bits of samplerate control value. */
    if ((USEX(FREQCTLH) & 15) != (audioPtr.curFctl >> 16)) {
	Disable();
	USEX(FREQCTLH) = (USEX(FREQCTLH) & ~15) | (audioPtr.curFctl >> 16);
	Enable();
    }
#endif

    #ifdef RECORDER
      #ifdef UART_BUFFERED
        if (UartFill() >= 1) {
            /* All commands are 3 bytes long. */
            register int c = UartPeekByte();
            if (c == 'D' /*0x44 bb bb*/) {
                if (UartFill() >= 3) {
                    register int arg;
                    UartGetByte();
                    arg = UartGetByte();
                    arg = (arg<<8) | UartGetByte();
                    /* Set date */
                    created[0] = arg;
                    putch('d');
                }
            } else if (c == 'T' /*54 bb bb*/) {
                if (UartFill() >= 3) {
                    register int arg;
                    UartGetByte();
                    arg = UartGetByte();
                    arg = (arg<<8) | UartGetByte();
                    /* Set time */
                    created[1] = arg;
                    putch('t');
                }
            } else if (c == 'S' /*53 bb bb*/) {
                if (UartFill() >= 3) {
                    register int arg;
                    UartGetByte();
                    arg = UartGetByte();
                    arg = (arg<<8) | UartGetByte();
                    /* Set seconds */
                    created[2] = arg;
                    putch('s');
                }
            } else {
                UartGetByte();
                /* unknown command */
                putch('u');
            }
        }
      #endif
    #endif/*RECORDER*/
    
    
    #ifdef SCI_UI
      /* play pause/continue */
      if ((USEX(SCI_AICTRL3) & CTRL3_PAUSE_ON)) {
          if (spiState != spi_pause) {
              oldSpiState = spiState;
              #ifdef UART_UI
                putch('p');//USEX(UART_DATA) = 'p';
              #endif
              spiState = spi_pause;
  
              stream_save = stream_wr_pointer;
              stream_wr_pointer = stream_rd_pointer+1;
              /* Does not matter if it is one off..
                 StreamDiff() returns the right value regardless. */
          }
      } else {
          if (spiState == spi_pause) {
          #ifdef UART_UI
                putch('c');//USEX(UART_DATA) = 'c';
          #endif
              spiState = oldSpiState;
              stream_wr_pointer = stream_save;
          }
      }
      #ifdef RECORDER
        if ((USEX(SCI_AICTRL3) & CTRL3_RECORD_ON)) {
            if (spiState != spi_waitima && spiState != spi_record) {
                record.record = 0x1234;
                goto out_of_wav;
            }
        } else {
            if (spiState == spi_waitima) {
                USEX(SCI_AICTRL0) = 0;
                goto out_of_wav;
            }
        }
      #endif/*RECORDER*/
    
      if ((USEX(SCI_AICTRL0) & 0x8000U)) {
        /* new song */
        USEX(SCI_AICTRL0) &= 0x7fff; /* ack command */
        /*NEW: clear pause and file ready bits automatically */
        USEX(SCI_AICTRL3) &= ~(CTRL3_PAUSE_ON | CTRL3_FILE_READY | CTRL3_AT_END);
        SpiSendClocks();

        #ifdef RECORDER
      out_of_wav:
          if (spiState == spi_waitima) {
              /* wait until MMC write finished */
              while ( SpiSendReceiveMmc(-1/*0xffffU*/, 16) != 0xffffU)
                  ;
  
              SpiSendClocks();
              Disable(); //prevent modu_int from trashing the FAT
              /* In vs1063 the stream buffer is used as SRC_BUFFER. */
  
              mmc.fileSize = spix.size;
	      /* Set the suffix. The suffix can be decided here because
		 FatFindDirEntry ignored suffix, so the name will be
		 unique regardless of the suffix. */
	      {
		register u_int16 fmt = (USEX(SCI_AICTRL3) & 0x00f0);
		if (fmt == AICTRL3_MP3) {
		  mmc.fileName[4] = ('M'<<8) | 'P';
		  mmc.fileName[5] = ('3'<<8);
		} else if (fmt == AICTRL3_VORBIS) {
		  mmc.fileName[4] = ('O'<<8) | 'G';
		  mmc.fileName[5] = ('G'<<8);
		} else {
		  mmc.fileName[4] = ('W'<<8) | 'A';
		  mmc.fileName[5] = ('V'<<8);

		  /* Fix WAV size fields, but only for WAV! */
		  ReadDiskSectorTo(mmc.freeStart, stream_buffer);
		  {
		    register u_int32 t = mmc.fileSize - 8;
		    stream_buffer[2] = SwapWord(t);
		    stream_buffer[3] = SwapWord(t>>16);
		    t -= 0x28;
		    stream_buffer[22] = SwapWord(t);
		    stream_buffer[23] = SwapWord(t>>16);
		  }
		  /* We don't care if write fails or not. */
		  MmcWrite(mmc.freeStart, stream_buffer);
		}
	      }
	      /* FsFatMakeFile() creates the directory entry. */
//puts("Making file");
	      FsFatMakeFile();

          } else {
              /* especially needed for low-bitrate!! */
              //USEX(SCI_MODE) |= (1<<SCIMB_CANCEL);
              //spiState = spi_zeros;
              /* Discard any unread data, including CRC, before starting again */
              {
                  register int i;
                  for (i=0; i<256+2; i++)
                      SpiSendReceiveMmc(-1/*0xffff*/,16);
              }
#if 1
	      /*wdog is still required! -- but this branch is not active
		because there is no sci-controlled recorder now. */
	      clockX = 2;
	      SetRate(10);
	      USEX(WDOG_CONFIG) = 1;
	      USEX(WDOG_RESET) = WDOG_RESET_VAL;
	      while (1)
		  ;
#endif
          }
          #ifdef UART_UI
            putch('e');//USEX(UART_DATA) = 'e';
          #endif
          /* Jump to new track immediately */
          MyReset();

        #else/*RECORDER*/
        
          /* Discard any unread data, including CRC, before starting again */
          {
              register int i;
              for (i=0; i<256+2; i++)
                  SpiSendReceiveMmc(-1/*0xffff*/,16);
          }
          /* Jump to new track immediately */
          MyReset();

        #endif/*elseRECORDER*/
      }
    #endif/*SCI_UI*/
    
    // Definitions for Standalone Legacy Keys (VS1011/VS1003/VS1053) SW1, SW2, SW3
    // These are read using special diode connections to the spi bus
      
    #define SW1 0xffffU
    #define SW2 0x8000U
    #define SW3 0x7fffU
    //#define SW4 0x4000U
      
    #ifdef ENABLE_UI
        
      // PKP: -- UI: USER INTERFACE --
         
      if (ui.Strobe) { // PKP: It's time to check the buttons and do UI calculations
	  /*
	    + SCLK = GPIDATA8
	    + XCS  = GPIDATA9
	    + SI   = GPIDATA10
	    + XDCS = GPIDATA11
	    The following code is compatible with the prototyping board buttons.
	    Code is currently 1 word longer.
	    Connecting the buttons differently 4 keys could be read directly.
	  */
	  ui.Data = ui.Strobe = 0;
                  
	  // PKP: -- UI: READ KEYS --
            
          #ifdef COMPAT_KEYS // PKP: Read the legacy 3-button keys
	    if (!(USEX(GPIO_IDATA) & (1<<10)/*SI*/)) {
		ui.Data = SW2;/*prev/vol-*/
	    } else if (!(USEX(GPIO_IDATA) & (1<<11)/*XDCS*/)) {
		ui.Data = SW1;/*next/vol+*/
	    } else {
		register u_int16 t = USEX(GPIO_ODATA);
		USEX(GPIO_ODATA) = t & ~SPI_xCS;
		/* Should have a short wait here, but seems to work this way.*/
		if (!(USEX(GPIO_IDATA) & (1<<11)/*XDCS*/)) {
		    ui.Data = SW3;/*play/pause/ etc*/
		} else if (!(USEX(GPIO_IDATA) & (1<<9)/*XCS*/)) {
		    ui.Data = SW3;
		}
		USEX(GPIO_ODATA) = t;
	    }
          #else // PKP: Read directly connected keys
	    //USEX(UART_DATA) = USEX(GPIO_IDATA) >> 4;
	    if (!(USEX(GPIO_IDATA) & (1<<10)/*SI*/)) {
		ui.Data = SW2;/*next*/
	    } else if (!(USEX(GPIO_IDATA) & (1<<11)/*XDCS*/)) {
		ui.Data = SW1;/*prev*/
	    } else if (!(USEX(GPIO_IDATA) & (1<<9)/*XCS*/)) {
		ui.Data = SW3;/*pause/play*/
	    } else if (!(USEX(GPIO_IDATA) & (1<<8)/*SCLK*/)) {
		ui.Data = SW1;/*prev*/
//	    } else if ((USEX(GPIO_IDATA) & (1<<4)/*GPIO4*/)) {
//		ui.Data = SW2;
	    }
          #endif
      
	  // PKP: -- UI: KEY FUNCTIONS --
      
          #ifdef RECORDER/*recorder with buttons*/


	    if (spiState == spi_waitima && ui.Data) {
		/* If key pressed in record mode, end record mode. */
		if (record.record == 0) {
		    /* End recording nicely to create valid files. */
		    record.record = 1;
		    USEX(SCI_MODE) |= (1<<SCIMB_CANCEL);
//putch('1');
		}
	    } else

	    if (ui.Data == ui.Key) { //PKP: current keystate is same as previous keystate
		if (ui.KeyTime < 12) {
		    ui.KeyTime++;
		} else if (ui.Key) {
		    register u_int16 vol = USEX(SCI_VOL);
		    if (ui.Key == SW1) {
			if (vol) {
			    vol -= 0x0101;
			    goto setvol;
			}
		    } else if (ui.Key == SW2) {
			if (vol != 0xffffU)
			    vol += 0x0101;
		    setvol:
			USEX(SCI_VOL) = vol;
                        #ifdef SAVE_VOLUME
			  volChanged = 1;
                        #endif/*SAVE_VOLUME*/
		    } else if (ui.Key == SW3) {
			if (ui.KeyTime == 12) {
			    if (spiState != spi_waitima) {
				/* SW3 long press,
				   goto record if not already */
				record.record = 0x1234;
//putch('R');
#if 1
				/* Discard any unread data, including CRC,
				   before starting again */
				{
				    register int i;
				    for (i=0; i<256+2; i++)
					SpiSendReceiveMmc(-1/*0xffff*/,16);
				}
				SpiSendClocks();

#if 0
				/*
				  Full reset is still required
				  either before or after encoding!
				  We now have it at the end or encoding
				  to reset all peripherals properly.
				*/
				clockX = 2;
				SetRate(10);
				USEX(WDOG_CONFIG) = 1;
				USEX(WDOG_RESET) = WDOG_RESET_VAL;
				while (1)
				    ;
#else
				MyReset();
#endif

#endif
			    }
                            /* If in record mode, end record mode. */
			    ui.KeyTime = -32767; /*one-shot*/
			    goto out_of_wav;
			}
		    }
		}
	    } else {
		if (ui.Data == 0) {
		    if (ui.KeyTime > 0 && ui.KeyTime < 12) {
			/* was pressed < 0.75 seconds */
			if (ui.Key == SW3) {
                                /* play pause/continue */
			    if (spiState == spi_waitima) {
				/* recording on, stop recording! */
#if 0
				putch('e');
				putch('1');
#endif
				goto out_of_wav;
			    } else {
				if (spiState != spi_pause) {
				    oldSpiState = spiState;
				    spiState = spi_pause;
        
				    stream_save = stream_wr_pointer;
				    stream_wr_pointer = stream_rd_pointer+1;
				    /* Does not matter if it is one off..
				       StreamDiff() returns the right value
				       regardless. */
				} else {
				    spiState = oldSpiState;
				    stream_wr_pointer = stream_save;
				}
			    }
			}
			if (ui.Key == SW1 || ui.Key == SW2) {
			    if (ui.Key == SW1) {
				USEX(SCI_AICTRL0) += 1; /* next file */
			    } else {
				/* Previous file */
				if (USEX(SCI_DECODE_TIME) < 5) {
				    if (USEX(SCI_AICTRL0) == 0) {
					USEX(SCI_AICTRL0) = USEX(SCI_AICTRL1) - 1;
				    } else {
					USEX(SCI_AICTRL0) -= 1; /* previous file */
				    }
				}
			    }
			out_of_wav:
			      if (spiState == spi_waitima) {
				  USEX(SER_DREQ) = 0;
				  /* wait until MMC write finished */
				  while ( SpiSendReceiveMmc(0xffffU, 16) != 0xffffU)
				      ;
				  SpiSendClocks();
				  Disable(); //prevent modu_int from trashing the FAT
				  /* Well, it would not because in VS1053 the stream
				     buffer is not used in encode mode.
				     But in vs1063 it again is (SRC_BUFFER). */
				  
				  mmc.fileSize = spix.size;
				  /* Set the suffix. The suffix can be decided
				     here because FatFindDirEntry ignored
				     suffix, so the name will be unique
				     regardless of the suffix. */
				  {
				    register u_int16 fmt =
					(USEX(SCI_AICTRL3) & 0x00f0);
				    if (fmt == AICTRL3_MP3) {
				      mmc.fileName[4] = ('M'<<8) | 'P';
				      mmc.fileName[5] = ('3'<<8);
				    } else if (fmt == AICTRL3_VORBIS) {
				      mmc.fileName[4] = ('O'<<8) | 'G';
				      mmc.fileName[5] = ('G'<<8);
				    } else {
				      mmc.fileName[4] = ('W'<<8) | 'A';
				      mmc.fileName[5] = ('V'<<8);

				      /*Fix WAV size fields */
				      ReadDiskSectorTo(mmc.freeStart,
						       stream_buffer);
				      {
					  register u_int32 t =
					      mmc.fileSize - 8;
					  stream_buffer[2] = SwapWord(t);
					  stream_buffer[3] = SwapWord(t>>16);
					  t -= 0x28;
					  stream_buffer[22] = SwapWord(t);
					  stream_buffer[23] = SwapWord(t>>16);
				      }
				      /*We don't care if it fails or not.*/
				      MmcWrite(mmc.freeStart, stream_buffer);
				    }
				  }
				  /* FsFatMakeFile() creates the directory
				     entry. */
//puts("Making file");
				  FsFatMakeFile();
                                #if 1
				  USEX(SCI_AICTRL0) = USEX(SCI_AICTRL3) = 0;
                                #endif

				  USEX(SCI_VOL) = playbackVol;
#if 1
				  USEX(SRC_CONTROL) = 0;
				  USEX(DECIM_CONTROL) =
				      DECIM_MODU2_PD | DECIM_MODU1_PD;

				  /* Full reset required before or after
				     encoding for it to work more than once.
				     TODO: find out why? DECIM-swap-channels?
				     Well, it's better because the encoding
				     patch should never return anyway.
				  */
				  clockX = 2; /*1.0x clock required for wdog*/
				  SetRate(10);
				  USEX(WDOG_CONFIG) = 1;
				  USEX(WDOG_RESET) = WDOG_RESET_VAL;
				  while (1)
				      ;
#else
				  MyReset();
#endif
			      }
        
                              /* especially needed for low-bitrateI!! */
			      USEX(SCI_MODE) |= (1<<SCIMB_CANCEL);
			      spiState = spi_zeros;

			      /* Discard any unread data, including CRC,
				 before starting again */
			      {
				  register int i;
				  for (i=0; i<256+2; i++)
				      SpiSendReceiveMmc(-1/*0xffff*/,16);
			      }
                              SpiSendClocks(); //raise chip select
			}
		    }
		}
		ui.Key = ui.Data;
		ui.KeyTime = 0;
	    }
	    //ui.Data = 0;
        
        
	    // PKP: Keys
        
	    // SW1 Short Press: Next Song
	    // SW2 Short Press: Previous Song
	    // SW3 Short Press: Pause/Play
        
	    // SW1 Long Press: Volume Up
	    // SW2 Long Press: Volume Down
	    // SW3 Long Press when Playing: Toggle Loudness
	    // SW3 Long Press when Paused: Toggle Shuffle Play
        
        
          #else /* start of non-RECORDER */
	    if (ui.Data == ui.Key) { // PKP: Current Keystate is same as previous Keystate
		if (ui.KeyTime < 8) {
		    ui.KeyTime++;
		} else // PKP: keystate has been consistent for 8 ticks - it's a long keypress
		if (ui.Key
                  #ifdef WAIT_FOR_BUTTON
		    && ui.KeyTime > 0
                  #endif
		    ) {
		    /* long presses */
		    register u_int16 vol = USEX(SCI_VOL);
		    if (ui.Key == SW1) {
			if (vol) {
			    vol -= 0x0101;
			    goto setvol;
			}
		    } else if (ui.Key == SW2) {
			if (vol != (s_int16)0xffffU)
			    vol += 0x0101;
		    setvol:
			USEX(SCI_VOL) = vol; /*in VS1011E disables ints! */
                        #ifdef SAVE_VOLUME
			  volChanged = 1;
                        #endif/*SAVE_VOLUME*/
		    } else {
			/* SW3 long press */
			if (ui.KeyTime == 8) {
			    ui.KeyTime = 9;
                            #if defined(LOUDNESS_ALWAYS_ON_OFF)
			      /* if pause on -- toggle random play */
			      USEX(SCI_AICTRL3) ^= CTRL3_RANDOM_PLAY;
			      if (spiState == spi_pause) {
				  spiState = oldSpiState;
				  stream_wr_pointer = stream_save;
			      }
                            #else /*LOUDNESS_ALWAYS_ON_OFF*/
                                
				 if (spiState
                                   #ifdef SW3_RANDOM_PLAY
				     !=
                                   #else
                                     ==
                                   #endif
				     spi_pause) {
				     /* if pause on -- toggle random play */
				     USEX(SCI_AICTRL3) ^= CTRL3_RANDOM_PLAY;
				 } else {
				     /* toggle loudness - have LSb set when ON */
				     USEX(SCI_BASS) ^= USEX(SCI_AICTRL2);
				 }
                              #endif/*else LOUDNESS_ALWAYS_ON_OFF*/
			}
		    }
		}
	    } else { // PKP: Keystate is not the same as previous keystate
		// PKP: Handle short keypresses
		if (ui.Data == 0
                  #ifdef WAIT_FOR_BUTTON
                    && ui.KeyTime > 0
                  #endif
		    ) {
		    // PKP: Handle end of short keypress (current keystate ui.Data=0)
		    /* short presses */
		    if (ui.KeyTime < 8) {
			/* was pressed < 0.5 seconds */
        
			if (ui.Key == SW3) {
			    /* play pause/continue -- not very good for midi.. */
			    if (spiState != spi_pause) {
                              #ifndef UI_PAUSE_BEFORE_PLAY
				oldSpiState = spiState;
				spiState = spi_pause;
				stream_save = stream_wr_pointer;
				stream_wr_pointer = stream_rd_pointer+1;
                              #endif/*!UI_PAUSE_BEFORE_PLAY*/
			    } else {
				spiState = oldSpiState;
				/* restore 'stolen' data */
				stream_wr_pointer = stream_save;
			    }
			} else if (ui.Key == SW1 || ui.Key == SW2) {
			    register u_int16 song = USEX(SCI_AICTRL0);
			    if (ui.Key == SW1) {
				song++; /* next file */
			    } else
                            #ifndef UI_PAUSE_BEFORE_PLAY
			      if (USEX(SCI_DECODE_TIME) < 5)
                            #endif
			      {
				  if (song == 0) {
				      song = USEX(SCI_AICTRL1)-1;
				  } else { /* previous file */
				      song--;
				  }
				  /* else the same file */
			      }
			    USEX(SCI_AICTRL0) = song;
        
                            /* especially needed for low-bitrate!! */
			    USEX(SCI_MODE) |= (1<<SCIMB_CANCEL);
			    spiState = spi_zeros;
			}
		    }//(ui.keyTime was < 8)
		}
		ui.Key = ui.Data;
		ui.KeyTime = 0;
	    }
        
            #if defined(SAVE_VOLUME)
	      /*Save volume when volume button is released. */
	      if (ui.Data == 0 &&
		  volChanged &&
		  spiState == spi_seek) {
		  volChanged = 0;
		  SaveWord(26/*volume=26*/, USEX(SCI_VOL));
	      }
            #endif/*SAVE_VOLUME*/

            //ui.Data = 0;
        #endif /* !RECORDER */
      }
    #endif /*USE_UI*/
    
    #ifdef PLAYTIME
      if (USEX(SCI_DECODE_TIME) > PLAYTIME && spiState != spi_zeros) {
          USEX(SCI_AICTRL0)++;
	  USEX(SCI_MODE) |= (1<<SCIMB_CANCEL);
          spiState = spi_zeros;
      }
    #endif/*PLAYTIME*/
    
    
    #ifdef RECORDER /* 88 words */
      if (spiState == spi_waitima) {
	  int z;
//putch('i');
//z = GetI6();
//putword(z);

	  /* spi_waitima: We are waiting for new data to arrive and
	     write data blocks when there is data available and the
	     card is ready for write. */
	  if (record.record == 3 && record.fill <= 0) {
//putch('o');
	      goto out_of_wav;
	  }
          if (SpiSendReceiveMmc(-1/*0xffffU*/, 16) == 0xffffU) {
              /* MMC Ready. */
              SpiSendClocks(); // extra clocks with chip select high
              SpiSendClocks(); // extra clocks with chip select high
  
              /* Note: byteswap is no longer required.

		 Because SD cards take longer time to write sectors
		 when they switch from one erase block to another,
		 we need quite a long FIFO for the audio data.
		 The FIFO in the encoding routine is not enough,
		 so we 'receive' data from the encoder's FIFO and
		 store it into our bigger FIFO and save it from there.
	      */
	      /* Data in second FIFO or last sector to write. */
	      if (record.fill >= 512 || record.record == 3) {
		  /* Save a sector from the second FIFO */
//putch('W');
                  WritePhysicalSectorY(spix.sector<<hcShift,
				       (__y void *)(YBUF+record.rd));
  
                  spix.sector++;
		  if (record.fill < 512) {
		      spix.size += record.fill; /* in bytes! */
		      record.fill = 0;
		  } else {
		      spix.size += 512; /* in bytes! */
		      record.fill -= 512;
		  }
                  record.rd += 256;
                  if (record.rd >= YBUFSZ) {
                      record.rd -= YBUFSZ;
//putch('<');
		  }
//putch('.');
              }
	      /* Limit filesize to about 2GB or to the available space. */
              if (spix.size >= 0x7ffffe00 || /*max size about 1 day at 24kHz*/
                  spix.size/512 >= mmc.freeSize ) {
                  /* disk full, end recording. */
                  goto out_of_wav;
              }
          }
  
          /* Fill second FIFO from encoder's FIFO. */
#define READ_SIZE 32 /*in words -- must be >= 22 words (size of WAV header used) and a power of 2 */
	  if (record.fill < YBUFSZ - READ_SIZE) {
              s_int16 imaFil = (bssWp - bssRp);
	      s_int16 read_size = READ_SIZE;
	      if (imaFil < 0)
		  imaFil += bssSize;
	      if (record.record == 1 &&
		  (USEX(SCI_MODE) & (1<<SCIMB_CANCEL)) == 0) {
		  if (imaFil < read_size) {
		      read_size = imaFil;
		      record.record = 2;
//putch('2');
		  }
	      }

              if (imaFil && imaFil >= read_size) {
                  s_int16 ne = record.wr + read_size;
                  if (ne >= YBUFSZ) 
                      ne -= YBUFSZ;
		  //if (ne != record.rd)
		  {
//putch('r');
                        memcpyYY(YBUF+record.wr, bssRp, read_size);
                        bssRp += read_size;
                        if (bssRp >= bssEnd) {
                            bssRp -= bssSize;
                        }
                        record.wr = ne;
                        record.fill += 2*read_size;
                  }
                  #ifndef SCI_UI
		  /* because encoder does not use WmaStereoCopy(),
		     applAddr does not get called, so we need to
		     update time base to trigger user interface in
		     recording mode. */
                    ui.Cnt += READ_SIZE; /* Mono linear 16-bit */
                    if (ui.Cnt >= USEX(SCI_AICTRL0)/*RECORD_FS*/ /16) {
                        ui.Cnt -= USEX(SCI_AICTRL0)/*RECORD_FS*/ /16;
                        ui.Timer++;
                        ui.Strobe = 1;
                    }
                  #endif
              }
	      if (record.record == 2) {
		  if ((s_int16)parametric_x.endFillByte < 0) {
		      *(__y u_int16 *)(YBUF+record.wr) =
			  parametric_x.endFillByte << 8;
		      record.fill += 1;
		      parametric_x.endFillByte = 0;
//putch('o');
//putch('d');
//putch('d');
		  }
		  record.record = 3;
//putch('3');
	      }
          }
  
          #if defined(LOWER_LOOPBACK_VOL) /* lower loopback volume */
            volume[0] = LOWER_LOOPBACK_VOL;
            volume[1] = LOWER_LOOPBACK_VOL;
          #endif
      } else if (spiState == spi_record) {
            record.fill = record.wr = 0;
            spix.size = 0; /* calculate file size, in bytes! */
            /* The WAV header is part of the data in vs1063. */
            spix.sector = mmc.freeStart;
            spiState = spi_waitima;
      }
    #endif/*RECORDER*/
    
    if (spiState == spi_file) { /* Play file from the beginning. */
        /* extra byte will be read if the file size is odd */
        spix.size = (fatInfo.fileSize+1)/2; /* convert to words */
        spiState = spi_seek;
setfrag:
        spix.fragment = &fragments[0];
        spix.sector   = fragments[0].start /*& 0x7fffffffUL*/;
        spix.fragSize = fragments[0].size; /*todo: *fatInfo.sectorsPerCluster*/
    }
 checkseek:
    if (spiState == spi_seek) {

    #ifndef SCI_UI
	/* In case we are just trying to find something to play,
	   also read keys once in a while..
	   One problem though: this idle hook may not be called in that case.
	*/
	//if (USEX(SCI_HDAT1) == 0)
	{
	    ui.Cnt += 64;
	    if (ui.Cnt > hwSampleRate/16) {
		ui.Cnt -= hwSampleRate/16;
		ui.Timer++;
		ui.Strobe = 1;
	    }
	}
    #endif

	/* Send the SD/MMC read block command */
	/* TODO: multiple block read for faster transfers!
	   May be needed by 24-bit FLAC files. */
	MmcCommand(MMC_READ_SINGLE_BLOCK|0x40,
		   (spix.sector & 0x7fffffffUL) << hcShift);

        spiState = spi_waitdata;
        spiCnt = 0;
        #ifdef HDATCHECK
          if (++hdatCheck == HDATCHECK && USEX(SCI_HDAT1) == 0) {
              /* if file not detected at HDATCHECK/2 kB, skip the rest */
              goto end_of_file;
          }
        #endif
    }

    /*TODO: handle spi_waitdata and spi_data
      in timer interrupt to increase throughput for FLAC */
    /*
      We have sent the read block command and now check if the data
      is available to be read out.
     */
    if (spiState == spi_waitdata) {
	register u_int16 res = SpiSendReceiveMmc(0xff00,8);
	if (res == 0xfe) { /* 0xfe is the data start token */
	    spiState = spi_data; /* go to fetch mode */
	    spiCnt = 0;
	    NextSector();
	} else if (res == 0xff) { /* card is still fetching block */
	    if ((s_int16)++spiCnt < 0) {
		MyReset(); /* timeout, reset immediately */
	    }
	} else {
	    /* Otherwise we have Read Error, and data is not transferred */
	    goto end_of_file;
	}
    } else if (spiState == spi_data) { /* data transfer state */
	/* Loop until the whole file is read, or stream buffer is full,
	   or the next sector needs to be read. */
	while (spix.size > 0) {
	    register int i, j = 32;
	    register u_int16 *wr;
	    /* FLAC has larger stream buffer */
	    if (USEX(SCI_HDAT1) == (('f'<<8) | 'L')) {
		register s_int16 t = stream_wr_pointer - stream_rd_pointer;
		if (t < 0)
		    t += 0x1800;
		if (t > 0x1800/*STREAM_BUFFER_SZ*/-40)
		    break; /* stream buffer full, leave loop */
		wr = (void *)stream_wr_pointer;
		if (j > spix.size)
		    j = spix.size;
		for (i=0;i<j;i++) {
		    *wr++ = SpiSendReceiveMmc(-1/*0xffffU*/,16);
		    if (wr >= &stream_buffer[0x1800/*STREAM_BUFFER_SZ*/])
			wr = stream_buffer;
		}
	    } else {
		if (StreamDiff() >= STREAM_BUFFER_SZ-40)
		    break; /* stream buffer full, leave loop */
		wr = (void *)stream_wr_pointer;
		if (j > spix.size)
		    j = spix.size;
		for (i=0;i<j;i++) {
		    *wr++ = SpiSendReceiveMmc(-1/*0xffffU*/,16);
		    if (wr >= &stream_buffer[STREAM_BUFFER_SZ])
			wr = stream_buffer;
		}
	    }
	    stream_wr_pointer = (void *)wr;
	    spix.size -= j;//32;
	    spiCnt += 32;
	    if (spiCnt >= 256) {
		/* 256 words read, get next block in file */
		spiState = spi_seek;
		SpiSendReceiveMmc(-1/*0xffff*/,16); /* discard crc */
#if 0
		if (USEX(SCI_HDAT1) == (('f'<<8) | 'L')) {
		    goto checkseek;
		}
#endif
		break;
	    }
	    /* watchdog clearing here doesn't work with midi */
	}
  
        if (spix.size <= 0) {
end_of_file:
            SpiSendClocks(); /* raise card chip select and send some clocks */

            #ifdef SCI_UI
            if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) == CTRL3_LOOP_SONG){
                  /* restart the same file immediately */
                  spiState = spi_file;
                  #ifdef UART_UI
                    putch('l');//USEX(UART_DATA) = 'l';
                  #endif
              } else
            #endif/*SCI_UI*/
            {
            #ifdef SCI_UI
		/*NEW: pause after play */
		if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
		    CTRL3_PAUSE_AFTER_PLAY) {
		    USEX(SCI_AICTRL3) |= CTRL3_PAUSE_ON | CTRL3_AT_END;
		} else {
		    USEX(SCI_AICTRL0) += 1; /* next file */
		}
		spiState = spi_zeros;
            #else
		{
		    register int m = (USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK);
		    if (m == CTRL3_LOOP_SONG) {
			/* restart the same file immediately */
			spiState = spi_file;
		    } else if (m == CTRL3_PAUSE_AFTER_PLAY) {
			oldSpiState = spi_zeros;
			spiState = spi_pause;
		    } else {
#ifdef LOOP_FILES
			/* keep playing the same file */
#else
			USEX(SCI_AICTRL0) += 1; /* next file */
#endif
			spiState = spi_zeros;

			if (USEX(SCI_HDAT1) == (('f'<<8) | 'L'))
			    USEX(SCI_MODE) |= (1<<SCIMB_CANCEL);
		    }
		}
            #endif
            }
        }
        #ifdef NO_WMA /* reject WMA files */
            if (USEX(SCI_HDAT1) == (('W'<<8)|'M')) {
		USEX(SCI_AICTRL0) += 1; /* next file */
		goto goreset;//MyReset();
	    }
        #endif
        #ifdef NO_AAC /* reject AAC files */
	    {
		register u_int16 t = USEX(SCI_HDAT1);
		//0x4154(AT) 0x4144(AD) 0x4d34(M4)
		if (t == 0x4154 || t == 0x4144 || t == 0x4d34) {
		    USEX(SCI_AICTRL0) += 1; /* next file */
		    goto goreset;//MyReset();
		}
	    }
        #endif
    } else if (spiState == spi_zeros) { /* send zeros after the file */
        /* Workaround for WMA resync after stream ended! */
        parametric_x.resync = 0;
	/* FLAC files use a bigger stream buffer,
	   also requires 12k of endFillByte! */
	if (USEX(SCI_HDAT1) == (('f'<<8) | 'L')) {
	    register s_int16 t = stream_wr_pointer - stream_rd_pointer;
	    if (t < 0)
		t += 0x1800;
	    if (t < 0x1800/*STREAM_BUFFER_SZ*/-40) {
		register int i, z = parametric_x.endFillByte * 0x0101;
		register u_int16 *wr = (void *)stream_wr_pointer;
		for (i=0;i<32;i++) {
		    *wr++ = z;
		    if (wr >= &stream_buffer[0x1800/*STREAM_BUFFER_SZ*/])
			wr = stream_buffer;
		    ++spiCnt;
		    stream_wr_pointer = (void *)wr;
		}
	    }
	} else {
	    if (StreamDiff() < STREAM_BUFFER_SZ-40) {
		register int i, z = parametric_x.endFillByte * 0x0101;
		register u_int16 *wr = (void *)stream_wr_pointer;
		for (i=0;i<32;i++) {
		    *wr++ = z;
		    if (wr >= &stream_buffer[STREAM_BUFFER_SZ])
			wr = stream_buffer;
		    ++spiCnt;
		    stream_wr_pointer = (void *)wr;
		}
	    }
	}

	if (USEX(SCI_HDAT1) == (('f'<<8) | 'L') && spiCnt >= 256+6*1024) {
	    goto goreset;
	} else if (USEX(SCI_HDAT1) == 0 ||
		   (USEX(SCI_HDAT1) == 0x7665 /*"ve"*/ && spiCnt >= 256+2*64)
		   || spiCnt >= 256+2050) {
	    /* 2050 bytes */
goreset:
	    /* Discard any unread SD data, including CRC,
	       before starting again */
	    {
		register int i;
		for (i=0; i<256+2; i++)
		    SpiSendReceiveMmc(-1/*0xffff*/,16);
	    }
	    SpiSendClocks();

#ifdef UART_UI
	    putch('e');//USEX(UART_DATA) = 'e';
#endif
	    MyReset();
	}
    } else {
    }
    
    #ifdef SCI_UI
        USEX(SER_DREQ) = 0;
    #else
    
      #ifdef NO_DREQ_LED
	/* Save space by not updating DREQ. */
      #else /*NO_DREQ_LED*/
        /* 15 words */
        {
            register int t = 0;
    
            if ((ui.Timer & 0x18) == 0) {
                /* DREQ indicates random play status */
                  t = USEX(SCI_AICTRL3);  /* lowest bit is global random play*/
                  #ifdef LOUDNESS_ALWAYS_ON_OFF
                    /* indicate random play */
                  #else /*LOUDNESS_ALWAYS_ON_OFF*/
                  
                    if (spiState
                    #ifdef SW3_RANDOM_PLAY
                          ==
                    #else
                          !=
                    #endif
                        spi_pause)
                        t = USEX(SCI_BASS); /* has lowest bit set if enabled*/
                  #endif /*else LOUDNESS_ALWAYS_ON_OFF*/
            }
            USEX(SER_DREQ) = t;
        }
      #endif/*!NO_DREQ_LED*/
    #endif/*!SCI_UI*/
}



void NewSinTest(void);
auto int PatchFrame(void);
void AacSetRate(register __c1 u_int16 rate);/*Patches some AAC problems*/

/* Check for and correct channel swap (DECIM restart bug) */
void DecimCheck(void) {
    register u_int16 decim = USEX(DECIM_CONTROL);
    register u_int16 lim = 0, val;
    register u_int16 tries = 0;
#if 0
putch('d');
putch('c');
putch('i');
putch('m');
#endif
    decim |= DECIM_ENABLE | DECIM_MODU2_PD; /*right chn to powerdown*/
    lim = clockX * (512/2 * 16); /* 13 seems enough, we use 14 */
    if (USEX(SCI_STATUS) & (1<<SCIST_AD_CLOCK))
	lim <<= 1; /*3MHz mode, double the wait*/
 again:
    tries++;
    USEX(DECIM_CONTROL) = decim;
    /* Wait at least 14 samples .. */
    {
	register u_int16 i;
	for (i=0; i<lim; i++) { /*two cycles per loop*/
	    USEX(DECIM_DATA_LEFT);
	    USEX(DECIM_DATA_LEFT);
	}
    }
#if 1
    /*if left chn is min/max, wrong order*/
    val = USEX(DECIM_DATA_LEFT);
    if (val == 0x8000U || val == 0x7fffU) {
	if (tries < 0x40) {
	    USEX(DECIM_CONTROL) &= ~DECIM_ENABLE;
	    goto again;
	}
#if 0
	putch('g');
	putch('u');
	putch('p');
#endif
    }
#endif
    /* take the other channel out of powerdown */
    USEX(DECIM_CONTROL) = decim & ~DECIM_MODU2_PD;
#if 0
    putch('c');
    putch('h');
    putch('n');
    putch('O');
    putch('\n');
#endif
}

u_int16 sectorBuffer[512/2] = {0};
void flacStreamBufferRestore(void);

/*
  TODO: use a timer interrupt for a watchdog timer.
 */
auto void MyMain(register s_int16 __a1 fullReset) {
    u_int16 tmp;
#ifdef ENABLE_HIGHREF
    USEX(SCI_STATUS) |= (1<<SCIST_REFERENCE_SEL); /*1.65V*/
#endif


	USEX(SCI_CLOCKF) = CORE_CLOCK_3_5X;



  #ifdef UART_BUFFERED
    UartFill();
  #endif

    /* Make sure we have the right upper ROM visible. */
    USEX(VS1053_IROM4) &= ~(IROM4_ENABLE);
    USEX(SCI_HDAT1) = 0; /* clear so FLAC is not indicated. */
    /* Just in case, restore the normal SDI interrupt handler (1024 words). */
    flacStreamBufferRestore();
    /* clear mp4ASC, seems to be needed after FLAC. */
    memset((void *)0x2000, 0, 64);

  #if defined(SCI_UI)
    /* no files yet */
    USEX(SCI_AICTRL1) = 0;
  #endif

    /* For MMC/SD handling the SCI_NEWMODE must be set, and also
       the relevant GPIO pins configured correctly. */

//#define SD_POWER_ENABLE (1<<7)
#define SD_POWER_ENABLE 0
#define LINE_IN_ENABLE (1<<6)
    USEX(SCI_MODE) = SciModeOr() | (USEX(SCI_MODE) & ~(1<<SCIMB_CANCEL));
    USEX(GPIO_ODATA) = SPI_xCS|SPI_CLK|SPI_xCS2|SD_POWER_ENABLE;
    USEX(GPIO_DDR)   = SPI_xCS|SPI_CLK|SPI_xCS2|SD_POWER_ENABLE
    #if defined(ENABLE_I2S)
	/* Enable I2S pins */
	| 0xf0;
      USEX(I2S_CONTROL) = 0x0c
    #endif
	;

    /* InitHardware() configures clock/PLL according to SCI_CLOCKF,
       initializes stream and audio buffer pointers, sets default
       samplerate (8000Hz), clears HDAT0,HDAT1,DECODE_TIME,AUDATA,
       and sets the cosmetic chip ID in parametric_x.chipID .

       User should set uartByteSpeed 960 before the call to get the
       default 9600bps transfer rate. (vs1063)
    */
    hwSampleRate = 11; /*force, mem is only cleared at boot, not restart*/
    uartByteSpeed = 960; //for vs1063, this MUST be same as target speed/10
    
//puts("Hello, World1");

    InitHardware(); /*no longer clears memory in vs1053/63 so preserves const*/
    audioPtr.curFctl = audioPtr.newFctl = 0x15555; //correct for 8kHz 20111117

//puts("starting");
	//puts("Hello, World");


  #ifdef RECORDER
    record.record <<= 1; /* 0x1234 to 0x2468 */
  #endif

  #ifndef USE_DEBUG
    /* Enable UART RX interrupt to be able to jump to emulator.
       Allows reflashing without MMC/SD inserted. */
    USEX(INT_ENABLE) = (1<<INT_EN_RX);
  #endif

    /* If not powerup reset, re-enable analog immediately. */
    if (!fullReset) {
	USEX(SCI_STATUS) &= ~(1<<SCIST_APDOWN2);
    }

  #ifdef ENABLE_UI
    /* Calculates ui.Timer and ui.Strobe,
       does not need to call UserHook anymore */
    applAddr = ApplAddr;
  #endif/*ENABLE_UI*/

    /* Allow DREQ and enable interrupts, VERY IMPORTANT! */
    //USEX(INT_ENABLE) = 0;
    USEX(INT_GLOB_ENA) = 0;
    USEX(INT_GLOB_ENA) = 0;
    USEX(INT_GLOB_ENA) = 0;

    /* wake up MMC! -- If this takes too long, watchdog will bite. */

  #ifdef SAVE_POSITION
    /* Clear the current song number so it will reset between cards. */
    if (USEX(SCI_AICTRL0)) /* 9 words */
	SaveWord(28/*AICTRL0*/, 0); //3 words
  #endif


    /* Initialize MMC/SD card, required for both player and recorder. */
    {
        register s_int16 i, cmd;

	// Start the MMC/SD card
    tryagain: 

#if SD_POWER_ENABLE
	/* If required, power down MMC/SD to make it leave SD mode.
	   However, it should not be powered down at each tryagain,
	   because sometimes multiple passes are required to bring
	   the card into operating mode. */
	// MMC reset using SD_POWER_ENABLE
	USEX(GPIO_DDR) |= SD_POWER_ENABLE;
	USEX(SER_DREQ) = 0;
	{
	  u_int16 gpio0_save = USEX(GPIO_ODATA);
	  USEX(GPIO_ODATA) = 1; //all 0 except eecs
	  for (i=0; i<32000; i++) {
	    USEX(GPIO_ODATA) = 1; //all 0 except eecs
	  }
	  USEX(GPIO_ODATA) = gpio0_save | SD_POWER_ENABLE;
	}
#endif

	/* Send clocks to MMC/SD with chip select high
	   -- is required by most cards. */
	for (i=0; i<512; i++) {
	    SpiSendClocks();
	}

#ifdef USE_DEBUG
	puts("GO_IDLE");
#endif
        /* MMC Init, command GO_IDLE_STATE should return 0x01 if all is ok. */
        i = MmcCommand(MMC_GO_IDLE_STATE/*CMD0*/|0x40,0);
        if (i != 1)
            goto tryagain;//continue; /* No valid idle response */

#ifdef USE_DEBUG
	puts("SEND_IF_COND");
#endif
        cmd = MMC_SEND_OP_COND|0x40;
        if (MmcCommand(MMC_SEND_IF_COND/*CMD8*/|0x40, 0x00000122/*2.7-3.6V*/)
	    == 1) {
	    /* MMC answers: 0x05 SD answers: 0x01 */
	    /* Read the whole R7 response? */
	    SpiSendReceiveMmc(-1, 32);
	    cmd = 0x40|41; /* ACMD41 - SD_SEND_OP_COND */
	}

        /* MMC Wake-up call, set to Not Idle (mmc returns 0x00)*/
        //i = 0; /* i is 1 when entered .. but does not make the code shorter*/
        while (1) {
            register int c;
#ifdef USE_DEBUG
	    puts("SEND_OP_COND");
#endif
	    if (cmd == (0x40|41)) {
		MmcCommand(0x40|55/*CMD55*/,0);
		c = MmcCommand(cmd/*MMC_SEND_OP_COND|0x40*/, 0x40000000UL);
	    } else {
		c = MmcCommand(cmd/*MMC_SEND_OP_COND|0x40*/, 0);
	    }
            if (c == 0)
                break;
            if (++i >= 12000/*< 0*/ || c != 1) {
          #ifdef USE_DEBUG
		puts("again");
          #endif
                goto tryagain; /* Not able to power up mmc */
            }
        }

	hcShift = 9;
	/* PRETEC's 1G MMC does not seemd to handle CMD58 here! */
	/* Support HC for SD, but not for MMC! */
	if (cmd == (0x40|41) &&
	    MmcCommand(MMC_READ_OCR/*CMD58*/|0x40, 0) == 0) {
	    if (SpiSendReceiveMmc(-1, 16) & (1<<(30-16))) {
		/* OCR[30]:CCS - card capacity select */
		/* HighCapacity, change shift values */
		hcShift = 0;
	    }
	    SpiSendReceiveMmc(-1, 16);
	}

#ifdef RECORDER
	if (MmcCommand(MMC_SEND_CID/*CMD10*/|0x40, 0) == 0) {
	    register s_int16 *p = stream_buffer;
	    register int t = 3200;
	    while (SpiSendReceiveMmc(0xff00, 8) == 0xff) {
		if (t-- == 0)
		    goto tryagain;
	    }
	    for (i=0; i<8; i++) {
		*p++ = SpiSendReceiveMmc(-1, 16);
//puthex(p[-1]);
		/*
		       Man     Productname   serial#   date
		          App             rev        res    crc7+stop
		  4G:  1D 4144 0000000000 00 0000157A 0 06A E3
		  64M: 02 0000 53444D3036 34 40185439 0 C77 75
		       00 0011 1122223333 44 44555566 6 677 77

		  2G:  02 544D 5341303247 03 9C046901 0 08B 3D Kingston SD
		  1G:  1B 534D 3030303030 10 B1FFC22A 0 08A 59 Extrememory SD
		*/
	    }
	    if (stream_buffer[1] == mmc.cid[1] &&
		stream_buffer[2] == mmc.cid[2] &&
		stream_buffer[3] == mmc.cid[3] &&
		stream_buffer[4] == mmc.cid[4] &&
		stream_buffer[5] == mmc.cid[5] &&
		stream_buffer[6] == mmc.cid[6] &&
		stream_buffer[7] == mmc.cid[7]) {
		/* same card -- use cached information */
	    } else {
		/* different card -- clear all cached values */
		memsetY(&mmc, 0, sizeof(mmc));
		memcpyXY(&mmc.cid, stream_buffer, 8);
#if 0
		puthex(stream_buffer[0]);
		puthex(stream_buffer[1]);
		puthex(stream_buffer[2]);
		puthex(stream_buffer[3]);
		puthex(stream_buffer[4]);
		puthex(stream_buffer[5]);
		puthex(stream_buffer[6]);
		puthex(stream_buffer[7]);
		puts(" CHG!");
#endif
	    }
//puts("=CID");
	}
#endif

        
      #if 0
	/* Set Block Size of 512 bytes -- default for at least HC */
	/* Needed by MaxNova S043618ATA 2J310700 MV016Q-MMC */
	/* Must check return value! (MicroSD restart!) */
	if (MmcCommand(MMC_SET_BLOCKLEN|0x40, 512) != 0)
	    goto tryagain;
	/* All OK return */
      #endif
    }


    // Init File System (FAT)
    /* Detects cards without partition table */
    memset(stream_buffer, 0, 512/2);
    if ((tmp = InitFileSystem()) == 0) {

#if 0
	//puthex((fatInfo.totSize - fatInfo.dataStart) / fatInfo.fatSectorsPerCluster - 2);
	puthex(fatInfo.numClusters);
	puts(" clusters");
#endif

    #ifdef RECORDER
      #ifndef SCI_UI
	ui.Key = 0;
	ui.KeyTime = -32767; /*Ignore record button release*/
      #endif

	/* When recording is selected, record.record is set to 0x1234,
	   and the software is restarted. The initialization shifts
	   record.record, so it will be 0x2468 here if we should go
	   to record mode. */
        if (record.record == 0x2468) {
            /* If record mode, re-enable analog power+drivers immediately. */
            USEX(SCI_STATUS) &= ~((1<<SCIST_APDOWN1)|(1<<SCIST_APDOWN2));
            memset(&record, 0, sizeof(record));
            //record.record = 0;
            //record.rd = record.wr = 0;

	    /*
	      FatPrepareForSaving() goes through FAT allocation table and
	      finds the largest contiguous area. It also goes through the
	      root directory and finds a free directory entry and creates
	      a unique filename that is used later when the file entry is
	      created by FsFatMakeFile().

	      Currently it checks only the 8 character name, so the suffix
	      can be later changed according to the encoding format and
	      the name still remains unique.
	     */
//puts("prepare");
            FatPrepareForSaving(); /* search free space and a valid name */
            if (mmc.dirLine >= 0) { /* found free entry */
                spiState = spi_record;
                #ifdef UART_UI
                  putch('r');//USEX(UART_DATA) = 'r';
                #endif

#if 1
		clockX = 9; /*4.5x required for 44+k stereo recording */
		SetRate(hwSampleRate);
#endif
                /* Lower monitor volume during recording. */
		playbackVol = USEX(SCI_VOL);
		USEX(SCI_VOL) += 0x1010;

		USEX(SCI_AICTRL0) = RECORD_FS; /*set in standalone.h*/
		USEX(SCI_AICTRL1) = USEX(SCI_WRAM);     /* auto gain */
		USEX(SCI_AICTRL2) = USEX(SCI_WRAMADDR); /* max gain */
//		USEX(SCI_WRAMADDR) = 0xe000 + 64; //64kbit/sec
//		USEX(SCI_WRAMADDR) = 0x0005; //q 5

		/* GPIO6 is high -> stereo line */
#if defined(LINE_IN_ENABLE) && LINE_IN_ENABLE
		if ((USEX(GPIO_IDATA) & LINE_IN_ENABLE)) {
		    /* mode: mp3, joint stereo

		       Note: 48kHz sample rate with stereo is too much for
		             3.5x clock. Thus we set 4.5x in record mode.
		     */
		    USEX(SCI_WRAMADDR) = 0xe000 + 160; //160kbit/sec
		    //USEX(SCI_WRAMADDR) = 0xe000 + 320; //320kbit/sec
		    USEX(SCI_AICTRL3) = AICTRL3_MP3 | AICTRL3_STEREO;
		    USEX(SCI_AICTRL1) = 0x0400;     /* fixed 1.0x gain */
		    USEX(SCI_MODE) |= (1<<SCIMB_LINE); /* Use line input */
		} else
#endif
		{
		    /* mode: mp3, mono left channel(mic) */
		    USEX(SCI_WRAMADDR) = 0xe000 + 128; //128kbit/sec
		    USEX(SCI_AICTRL3) =
			AICTRL3_MP3
			//AICTRL3_VORBIS
			//AICTRL3_G722
			//AICTRL3_G711_ALAW
			//AICTRL3_G711_ULAW
			//AICTRL3_PCM
			//AICTRL3_ADPCM
#if 0
			| AICTRL3_MONOMIX /* 4: mono downmix */
#else
			| AICTRL3_LEFT /* 2: left channel, 3: right channel */
#endif
			;
		    USEX(SCI_MODE) &= ~(1<<SCIMB_LINE); /* Use mic input */
		}
		USEX(SCI_MODE) &= ~(1<<SCIMB_CANCEL);

//puts("recorder");
		//record.rd = record.wr = record.fill = 0;
		/*
		  NOTE: Encoders() sets applAddr to NULL, but on the other
		        hand it does not get called anyway because encoders
			do not use AudioOutputSamples().
		*/
//putch('e');
#if 1
                CallEncoders(); /* with patch. loop there until reset */
#else
                CallIROM4(Encoders); /* loop there until reset */
#endif
            }
        }
    #endif/*RECORDER*/

      #ifdef SCI_UI
	if ((USEX(SCI_AICTRL3) & CTRL3_NO_NUMFILES)) {
	  /* Startup is faster if we don't need to find number of files. */
	  USEX(SCI_AICTRL1) = 0x7fff;
	} else
      #endif/*SCI_UI*/
	{
	  /* determine number of playable files */
	  USEX(SCI_AICTRL1) = OpenFile(0xffffU);
	  //putch(USEX(SCI_AICTRL1)>>8);
	  //putch(USEX(SCI_AICTRL1));
	}
	
	if ((USEX(SCI_AICTRL3) & CTRL3_RANDOM_PLAY)) {
        #ifdef SHUFFLE_PLAY /*shuffle 23 words*/
	  USEX(SCI_AICTRL0) = Shuffle(USEX(SCI_AICTRL1), USEX(SCI_AICTRL0));
        #else
          /* random play (not shuffle!) */
	  USEX(SCI_AICTRL0) = (myrand() % USEX(SCI_AICTRL1));
        #endif
        }
        
	/* Open the file to play. The file number is in AICTRL0. */
        if (OpenFile(USEX(SCI_AICTRL0)
        #ifdef SCI_UI
                        &= 0x7fffU
                       //max 32768 files in SCI-controlled version
        #endif
            ) < 0) {
#if 0
	    if (fatInfo.IS_FAT_32) {
              puts("FAT32");
	    } else if (fatInfo.FilSysType == 0x3231) {
              puts("FAT12");
	    } else {
              puts("FAT16");
	    }
	    {
		int i=-1;
		do {
		    i++;
		    puthex(fragments[i].start>>16);
		    puthex(fragments[i].start);
		    putchar('/');
		    puthex(fragments[i].size);
		    putchar(' ');
		} while ((s_int32)fragments[i].start >= 0);
	    }
#endif



            #if defined(SKIP_ID3V2) /* 48 words */
              /* check for ID3v2 */
              /* An ID3v2 tag can be detected with the following pattern:
                 $49 44 33 yy yy xx zz zz zz zz */
              ReadDiskSectorTo(fragments[0].start, stream_buffer);
              if ((GetLong(0) & 0xffffffL) == 0x334449) {
                  /* skip in 16kB resolution, assume tags are smaller than 2MB */
                  /* 0.5kB resolution now */
                  register u_int16 skip = 32 * GetByte(7) + GetByte(8)/4;
                  /* guard against small fragments */
                  if (skip >= fragments[0].size)
                      skip = fragments[0].size-1;
                  fragments[0].start += skip;
                  fragments[0].size  -= skip;
                  fatInfo.fileSize -= (u_int32)skip * 512;
              }
            #endif
/*ADTS is found from FLAC files, and some of them crash the decoder */
	    #if 0 && defined(RECORDER)
	      ReadDiskSectorTo(fragments[0].start, stream_buffer);
	      if (GetLong(0) == 0x43614c66) {
		  /* Skip fLaC files in recorder */
		  USEX(SCI_AICTRL0)++;
		  MyReset();
	      }
	    #endif

            
	    #if defined(SCI_UI)
              if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
                  CTRL3_PAUSE_BEFORE_PLAY) {
                  /* Mark file ready and go to pause state in UserHook */
                  USEX(SCI_AICTRL3) |= CTRL3_PAUSE_ON;
              }
              /*NEW: set FILE_READY always */
              USEX(SCI_AICTRL3) = CTRL3_FILE_READY | 
                  (USEX(SCI_AICTRL3) & ~CTRL3_AT_END);
            #endif
            
            #if defined(PAUSE_AT_POWERON)
	      /*pause at power-on, then use pause after play*/
              if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
                  CTRL3_PAUSE_BEFORE_PLAY) {
                  USEX(SCI_AICTRL3) = CTRL3_PAUSE_AFTER_PLAY;
                  oldSpiState = spi_file;
                  spiState = spi_pause;
                  stream_save = stream_wr_pointer;
              } else {
                  spiState = spi_file;
              }
            #else /*PAUSE_AT_POWERON*/
            
              #if defined(UI_PAUSE_BEFORE_PLAY) && defined(ENABLE_UI)
                oldSpiState = spi_file;
                spiState = spi_pause;
                stream_save = stream_wr_pointer;
              #else
		if ((USEX(SCI_AICTRL3) & CTRL3_PLAY_MODE_MASK) ==
		    CTRL3_PAUSE_BEFORE_PLAY) {
		    oldSpiState = spi_file;
		    spiState = spi_pause;
		    stream_save = stream_wr_pointer;
		} else {
		    spiState = spi_file;
		}
              #endif
            #endif /*else PAUSE_AT_POWERON*/
            
            
            #ifdef HDATCHECK
              hdatCheck = 0;
            #endif
        } else {
        gofirst:
            USEX(SCI_AICTRL0) = 0; /* file not found, jump to first file */
            MyReset();
        }
    } else {
        /* Initialize again */
#if 0
	puthex(tmp);
	putchar('!');
	putchar('f');
	putchar('a');
	putchar('t');
	putchar(' ');
#endif
        goto gofirst;//MyReset();
    }

    #ifdef SAVE_POSITION
      /* Save current song number to EEPROM */
      SaveWord(28/*AICTRL0*/, USEX(SCI_AICTRL0));
    #endif
    
    USEX(INT_ENABLE) = ENABLE_SCI_DAC_RX;
    USEX(SCI_DECODE_TIME) = 0;

#if 0 //Sine Test
//      USEX(SCI_VOL) = 0;
      USEX(SCI_VOL) = 0x0404;
//      SetRate(44100U);
      SetRate(48000U);
      USEX(SCI_AICTRL0) = 2972; /*1kHz*/
      USEX(SCI_AICTRL1) = 2972;
      CallIROM4(NewSinTest);
#endif

    stream_rd_pointer = stream_wr_pointer = (void*)stream_buffer;
    stream_rd_odd = 0;

    /*
      For VS1063 we rewrote the main loop.
      UserCheckTag() and UserAudioFormats() can be eliminated here.
    */
    parametric_x.config1 = 0x0010; /*No implicit upsample -- VS1063 20110321*/
    parametric_x.resync = 32767; /* enable resyncs */
    {
	__y u_int32 newHead = 0;
	layer123y.fr.header_change = 3;
	while (1) {
#if 0
	    newHead = UserCheckTag(newHead);
	    if (UserAudioFormats(newHead) != 0) {
		/* User codec decoded something */
		newHead = 0;
	    } else 
#endif
	    if (CheckAudioFormats(newHead) != 0) {
		/* One of our codecs decoded something */
		newHead = 0; /* Decoded something, start again */
	    }
	    /* Skip a byte and try again. */
	    StreamHeadShift(&newHead);

	    /* Added: Calls user hook once per byte when searching for audio */
	    UserHook();
	}
    }
}
