/*
 * Copyright (C) 2013 by egnite GmbH
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

#include <dev/i2cbus.h>
#include <sys/timer.h>

#include "locmote.h"

/*!
 * \file locmote.c
 * \brief Nut/OS I2C slave driver to communicate with the ATtiny on the Locmote add-on board.
 *
 * \verbatim
 * $Id$
 * \endverbatim
 */

/*!
 * \brief I2C slave structure.
 *
 * Each hardware specific driver offers a global variable of this type,
 * which applications must pass to NutRegisterI2cSlave() to attach
 * a specific device to a specific bus.
 */
NUTI2C_SLAVE i2cLocmote = {
    NULL,   /*!< \brief Pointer to the bus driver, slave_bus. */
    35,     /*!< \brief Device's 7 bit slave address, slave_address. */
    100,    /*!< \brief Slave access timeout, slave_timeout. */
    NULL    /*!< \brief Transfer message, slave_msg. */
};

/*!
 * \brief Set Locmote control register.
 *
 * \param reg   The register to set, which may be any of the following:
 *              - LMC_ENA_REG enable register to set control bits to 1.
 *              - LMC_DIS_REG disable register to reset control bits to 0.
 * \param flags Value to write into the specified register.
 *
 * \return 0 on success or -1 in case of an error.
 */
int LocmoteSetRegister(uint8_t reg, uint8_t flags)
{
    int rc = 10;
    uint8_t tx[2];

    while (rc--) {
        tx[0] = reg;
        tx[1] = flags;
        if (NutI2cMasterTransceive(&i2cLocmote, tx, 2, NULL, 0) == 0) {
            rc = 0;
            break;
        }
    }
    return rc;
}

/*!
 * \brief Get Locmote control register contents.
 *
 * \param reg   The register to set, which may be any of the following:
 *              - LMC_TEMP_REG temperature register.
 *              - LMC_STA_REG  status register.
 *              - LMC_ENA_REG  control enable register, typically 0 when read.
 *              - LMC_DIS_REG  control disable register, typically 0 when read.
 *              - LMC_VOLT_REG voltage register.
 * \param flags Value to write into the specified register.
 *
 * \return 0 on success or -1 in case of an error.
 */
int LocmoteGetRegister(uint8_t reg)
{
    int rc = 10;
    uint8_t rx;

    while (rc--) {
        if (NutI2cMasterTransceive(&i2cLocmote, &reg, 1, &rx, 1) == 1) {
            rc = rx;
            break;
        }
    }
    return rc;
}

/*!
 * \brief Get Locmote status register.
 *
 * \return In case of an error, -1 is returned. Otherwise the return value
 *         contains the status bits:
 *         - LMC_IO3V_MSK: If set, Locmote I/O voltage is 3.3V.
 *         - LMC_IO5V_MSK: If set, Locmote I/O voltage is 5.0V.
 *         - LMC_IOOK_MSK: If set, Locmote I/O voltage is OK.
 *         - LMC_US2_MSK: If set, the SIM module is connected to the secondary UART.
 *         - LMC_GPS_MSK: If set, the UART is connected to the GPS module. Otherwise it is connected to the GSM module.
 *         - LMC_PWR_MSK: If set, the GSM/GPS module is powered.
 *         - LMC_KEY_MSK: If set, the (imaginary) power key is pressed.
 */
int LocmoteStatus(void)
{
    return LocmoteGetRegister(LMC_STA_REG);
}

/*!
 * \brief Get Locmote temperature.
 *
 * \return Rough estimate of the chip temperature in degrees Celsius,
 *         with a resolution of 4 degrees.
 */
int LocmoteChipTemperature(void)
{
    int rc;

    rc = LocmoteGetRegister(LMC_TEMP_REG);
    if (rc > 0) {
        rc <<= 2;
        rc -= 270;
    }
    return rc;
}

/*!
 * \brief Get Locmote I/O voltage.
 *
 * \return I/O voltage in mV, with a resolution of 25mV.
 */
int LocmoteIoVoltage(void)
{
    int rc;

    rc = LocmoteGetRegister(LMC_VOLT_REG);
    if (rc > 0) {
        //rc *= 99;
        //rc >>= 2;
        rc *= 22;
    }
    return rc;
}

/*!
 * \brief Set or clear Locmote control bit.
 *
 * After setting or clearing the bit, this function will check the status
 * for verification. If the status doesn't reflect the intended change,
 * up to 10 retries are done.
 *
 * \param bit Bit number to set, which may be any of the following values:
 *            - LMC_GPS_BIT connects UART to GPS module. Otherwise the GSM module is connected.
 *            - LMC_KEY_BIT set to 1 presses, set to 0 releases the imaginary power key.
 * \param val Value to set the specified bit to, either 0 or 1.
 *
 * \return 0 on success, or -1 if the bit couldn't be set after 10 retries.
 */
int LocmoteSetBit(uint8_t bit, int val)
{
    int retries = 10;
    int rc;

    do {
        rc = LocmoteSetRegister(val ? LMC_ENA_REG : LMC_DIS_REG, _BV(bit));
        if (rc < 0) {
            break;
        }
        rc = LocmoteStatus();
        if (rc < 0) {
            break;
        }
        if ((((uint8_t) rc & _BV(bit)) != 0) == val) {
            rc = 0;
            break;
        }
    } while (retries--);

    return rc;
}

/*!
 * \brief Switch power supply of Locmote's GSM/GPS module on or off.
 *
 * \param on Set to zero to switch off the GSM/GPS power supply.
 *           Any other value will switch it on.
 */
int LocmotePower(int on)
{
    int is_on;
    int rc;

    /* Make sure that the key is released. */
    rc = LocmoteSetBit(LMC_KEY_BIT, 0);
    if (rc == 0) {
        /* Query the current power status after a short delay. */
        NutSleep(10);
        rc = LocmoteStatus();
        /* Make sure that we got a valid status. -1 means invalid. */
        if (rc >= 0) {
            /* Check wether the GSM/GPS power is on or off. */
            is_on = (rc & LMC_PWR_MSK) != 0;
            if (is_on == on) {
                /* Power status is already as requested. */
                rc = 0;
            } else {
                /* Power status is not as requested. Like on a typical
                 * mobile we need to press a key until the power status
                 * changes from off to on or vice versa. So let's press
                 * the key. */
                rc = LocmoteSetBit(LMC_KEY_BIT, 1);
                if (rc == 0) {
                    /* Wait up to 5 seconds for status change. */
                    rc = 10;
                    while (rc--) {
                        int st;
                        /* Check the status after half a second. */
                        NutSleep(500);
                        st = LocmoteStatus();
                        is_on = st >= 0 && (st & LMC_PWR_MSK) != 0;
                        /* If the power status is as request, we are done. */
                        if (is_on == on) {
                            rc = 0;
                            break;
                        }
                    }
                    /* Release the key. */
                    LocmoteSetBit(LMC_KEY_BIT, 0);
                }
            }
        }
    }
    return rc;
}

/*!
 * \brief Enable Locmote.
 *
 * \param on Set to 1 to enable or to 0 to disable Locmote.
 */
void LocmoteEnable(int on)
{
    if (on) {
        sbi(PORTE, 3);
        sbi(DDRE, 3);
    } else {
        cbi(PORTE, 3);
        cbi(DDRE, 3);
    }
}

