/*
 * Copyright (C) 2013 by egnite GmbH
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "pducodec.h"

/*!
 * \file pducodec.c
 * \brief Encode and decode SMS PDU format.
 *
 * \verbatim
 * $Id$
 * \endverbatim
 */


/* Enable this to add SC information to outgoing messages.
   With most providers this is not required.
#define SERVICE_CENTER_ADDRESS  "491760000443"
*/

static char hexdigit[] = "0123456789ABCDEF";

/*!
 * \brief Convert hexadecimal character to binary value.
 *
 * \param c Character to convert.
 *
 * \return Binary value or just 0 if the character was not a hex digit.
 */
static uint8_t hex2bin(char c)
{
    if (c >= '0' && c <= '9')
        return c - '0';
    if (c >= 'A' && c <= 'F')
        return c - 'A' + 10;
    return 0;
}

/*!
 * \brief Decode single octet.
 *
 * \param pdu Pointer to a buffer containing the PDU formatted octets.
 * \param val Pointer to a variable that receives the decoded value.
 *
 * \return Pointer to the next octet in the given buffer.
 */
static const char *PduDecHexOctet(const char *pdu, uint8_t *val)
{
    *val = hex2bin(*pdu);
    ++pdu;
    *val <<= 4;
    *val |= hex2bin(*pdu);

    return ++pdu;
}

/*!
 * \brief Decode semi octet.
 *
 * \param pdu Pointer to a buffer containing the PDU formatted octets.
 * \param val Pointer to a variable that receives the decoded value.
 *
 * \return Pointer to the next octet in the given buffer.
 */
static const char *PduDecSemiOctet(const char *pdu, uint8_t *val)
{
    *val = hex2bin(*(pdu + 1));
    *val *= 10;
    *val += hex2bin(*pdu);

    return pdu + 2;
}

/*!
 * \brief Encode single octet.
 *
 * \param pdu Pointer to a buffer that receives the PDU format.
 * \param val Octet to encode.
 *
 * \return Pointer to the next octet in the given buffer.
 */
char *PduEncHexOctet(char *pdu, uint8_t val)
{
    *pdu++ = hexdigit[val >> 4];
    *pdu++ = hexdigit[val & 0x0F];

    return pdu;
}

/*!
 * \brief Decode semi-octet representation.
 *
 * \param pdu Pointer to a buffer containing the PDU formatted octets.
 *
 * \return Pointer to the next octet in the given buffer.
 */
const char *PduDecSemiOctetString(const char *pdu, int len, char *buf, int siz)
{
    int i;

    for (i = 0; i < len; i += 2) {
        *buf++ = *(pdu + 1);
        if (*pdu == 'F') {
            pdu += 2;
            break;
        }
        *buf++ = *pdu;
        pdu += 2;
    }
    *buf = '\0';

    return pdu;
}

/*!
 * \brief Encode semi-octet representation.
 *
 * \param pdu Pointer to a buffer that receives the PDU format.
 *
 * \return Pointer to the next octet in the given buffer.
 */
char *PduEncSemiOctetString(char *pdu, const char *num)
{
    while (*num) {
        pdu[1] = *num++;
        if (*num) {
            pdu[0] = *num++;
        } else {
            pdu[0] = 'F';
        }
        pdu += 2;
    }
    return pdu;
}

/*!
 * \brief Decode timestamp from PDU format.
 *
 * \param pdu Pointer to a buffer containing the PDU formatted octets.
 *
 * \return Pointer to the next octet in the given buffer.
 */
const char *PduDecTime(const char *pdu, time_t *t)
{
    uint8_t val;
    struct _tm ltm;

    pdu = PduDecSemiOctet(pdu, &val);
    ltm.tm_year = 100 + val;
    pdu = PduDecSemiOctet(pdu, &val);
    ltm.tm_mon = val - 1;
    pdu = PduDecSemiOctet(pdu, &val);
    ltm.tm_mday = val;

    pdu = PduDecSemiOctet(pdu, &val);
    ltm.tm_hour = val;
    pdu = PduDecSemiOctet(pdu, &val);
    ltm.tm_min = val;
    pdu = PduDecSemiOctet(pdu, &val);
    ltm.tm_sec = val;

    /* Time zone. */
    pdu += 2;
    *t = mktime(&ltm);

    return pdu;
}

/*!
 * \brief Decode data from PDU format.
 *
 * \param pdu Pointer to a buffer containing the PDU formatted octets.
 */
int PduDecData(const char *pdu, int len, int octets, uint8_t *msg)
{
    int n;
    int i;
    int j;
    uint8_t *data;
    uint8_t c;

    data = (uint8_t *) malloc(len);
    for (n = 0; n < len; n++) {
        pdu = PduDecHexOctet(pdu, &data[n]);
    }

    for (n = 0, i = 0; n < len; n++) {
        c = 0;
        for (j = 0; j < 7; j++, i++) {
            if (data[i >> 3] & (1 << (i & 7))) {
                c |= 1 << j;
            }
        }
        *msg++ = c;
    }
    *msg = '\0';
    free(data);

    return len;
}

/*!
 * \brief Encode data into PDU format.
 *
 * \param pdu Pointer to a buffer that receives the PDU format.
 */
char *PduEncData(const uint8_t *msg, int len, char *pdu, int octets)
{
    int n;
    int i;
    int b;
    uint8_t *data;
    uint32_t col = 0;

    data = (uint8_t *) calloc(len, 1);
    for (n = 0, i = 0, b = 0; n < len; n++) {
        col |= (msg[n] & 0x7f) << b;
        b += 7;
        while (b >= 8) {
            data[i++] = col & 0xff;
            col >>= 8;
            b -= 8;
        }
    }
    while (b > 0) {
        data[i++] = col & 0xff;
        col >>= 8;
        b -= 8;
    }
    for (n = 0; n < i; n++) {
        *pdu++ = hexdigit[data[n] >> 4];
        *pdu++ = hexdigit[data[n] & 0x0f];
    }
    return pdu;
}

/*!
 * \brief Decode PDU formatted SMS to be sent to service center.
 *
 * \param msg Pointer to the SMS information structure.
 */
const char *PduDecSmsSubmit(const char *pdu, SHORT_MESSAGE *msg)
{
    uint8_t len;

    /* Message reference number. */
    pdu = PduDecHexOctet(pdu, &len);
    if (len) {
    }

    /* Receiver address. */
    pdu = PduDecHexOctet(pdu, &len);
    if (len) {
        uint8_t nt;
        char *number = msg->msg_ms_addr;

        pdu = PduDecHexOctet(pdu, &nt);
        if (nt == 0x91) {
            *number++ = '+';
        }
        else if (nt != 0x81) {
            free(msg);
            return NULL;
        }
        pdu = PduDecSemiOctetString(pdu, len, number, sizeof(msg->msg_ms_addr) - 1);
        //printf("MS(%d) %s\n", len, msg->msg_ms_addr);
    }

    pdu = PduDecHexOctet(pdu, &msg->msg_proto);
    pdu = PduDecHexOctet(pdu, &msg->msg_enc);
    switch ((msg->msg_flags & TP_VPF) >> TP_VPF_LSB) {
    case TP_VPF_NONE:
        break;
    case TP_VPF_INTEGER:
        pdu = PduDecHexOctet(pdu, &len);
        break;
    case TP_VPF_OCTET:
        break;
    }

    pdu = PduDecHexOctet(pdu, &msg->msg_len);
    if (msg->msg_len) {
        msg->msg_data = (uint8_t *) malloc(msg->msg_len + 1);
        PduDecData(pdu, msg->msg_len, 0, msg->msg_data);
        //printf("MSG(%d): '%s'\n", msg->msg_len, msg->msg_data);
    }
    return pdu;
}

/*!
 * \brief Decode PDU formatted SMS from service center.
 *
 * \param msg Pointer to the SMS information structure.
 */
const char *PduDecSmsDeliver(const char *pdu, SHORT_MESSAGE *msg)
{
    uint8_t len;

    pdu = PduDecHexOctet(pdu, &len);
    if (len) {
        uint8_t nt;
        char *number = msg->msg_ms_addr;

        pdu = PduDecHexOctet(pdu, &nt);
        if (nt == 0x91) {
            *number++ = '+';
        }
        else if (nt != 0x81) {
            free(msg);
            return NULL;
        }
        pdu = PduDecSemiOctetString(pdu, len, number, sizeof(msg->msg_ms_addr) - 1);
        //printf("MS(%d) %s\n", len, msg->msg_ms_addr);
    }
    pdu = PduDecHexOctet(pdu, &msg->msg_proto);
    pdu = PduDecHexOctet(pdu, &msg->msg_enc);
    pdu = PduDecTime(pdu, &msg->msg_time);
#if 0
    {
        struct _tm *ltm;

        ltm = localtime(&msg->msg_time);
        printf("Time %02d:%02d:%02d\n", ltm->tm_hour, ltm->tm_min, ltm->tm_sec);
        printf("Date %02d.%02d.%04d\n", ltm->tm_mday, ltm->tm_mon + 1, ltm->tm_year + 1900);
    }
#endif
    pdu = PduDecHexOctet(pdu, &msg->msg_len);
    if (msg->msg_len) {
        msg->msg_data = (uint8_t *) malloc(msg->msg_len + 1);
        PduDecData(pdu, msg->msg_len, 0, msg->msg_data);
        //printf("MSG(%d): '%s'\n", msg->msg_len, msg->msg_data);
    }
    return pdu;
}

/*!
 * \brief Decode PDU formatted SMS.
 *
 * \return Pointer to an allocated SMS information structure. Must be
 *         released by the caller.
 */
SHORT_MESSAGE *PduDecMsg(const char *pdu, int to_sc)
{
    SHORT_MESSAGE *msg;
    uint8_t len;

    msg = (SHORT_MESSAGE *) calloc(1, sizeof(SHORT_MESSAGE));
    if (msg == NULL) {
        return NULL;
    }

    /* Decode optional SMSC information. */
    pdu = PduDecHexOctet(pdu, &len);
    if (len) {
        uint8_t nt;
        char *number = msg->msg_sc_addr;

        pdu = PduDecHexOctet(pdu, &nt);
        if (nt == 0x91) {
            *number++ = '+';
        }
        else if (nt != 0x81) {
            free(msg);
            return NULL;
        }
        pdu = PduDecSemiOctetString(pdu, (len - 1) << 1, number, sizeof(msg->msg_sc_addr) - 1);
        //printf("SC(%d) %s\n", len, msg->msg_sc_addr);
    }

    /* Decode message type. */
    pdu = PduDecHexOctet(pdu, &msg->msg_flags);

    if (to_sc) {
        /* From MS to SC. */
        switch((msg->msg_flags & TP_MTI) >> TP_MTI_LSB) {
        case TP_MTI_DELIVER_REPORT:
            break;
        case TP_MTI_SUBMIT:
            pdu = PduDecSmsSubmit(pdu, msg);
            break;
        case TP_MTI_COMMAND:
            break;
        }
    } else {
        /* From SC to MS. */
        switch((msg->msg_flags & TP_MTI) >> TP_MTI_LSB) {
        case TP_MTI_DELIVER:
            pdu = PduDecSmsDeliver(pdu, msg);
            break;
        case TP_MTI_SUBMIT_REPORT:
            break;
        case TP_MTI_STATUS_REPORT:
            break;
        }
    }
    return msg;
}

/*!
 * \brief Encode SMS into PDU format.
 *
 * \param msg Pointer to the SMS information structure.
 *
 * \return Allocated buffer containing the PDU format. The caller must
 *         release this buffer.
 */
char *PduEncMsg(const SHORT_MESSAGE *msg)
{
    char *pdubuf;
    char *pdu;
    int atype = msg->msg_ms_addr[0] == '+';

    pdubuf = (char *) malloc(512);
    pdu = pdubuf;

    /* Service center information. */
#if defined(SERVICE_CENTER_ADDRESS)
    pdu = PduEncHexOctet(pdu, 7);
    pdu = PduEncHexOctet(pdu, 0x91);
    pdu = PduEncSemiOctetString(pdu, SERVICE_CENTER_ADDRESS);
#else
    pdu = PduEncHexOctet(pdu, 0);
#endif
    /* Message type. */
    pdu = PduEncHexOctet(pdu, msg->msg_flags);
    /* Message reference number. */
    pdu = PduEncHexOctet(pdu, 0);
    /* Destination address length. */
    pdu = PduEncHexOctet(pdu, (uint8_t) strlen(msg->msg_ms_addr) - atype);
    /* Destination address type. */
    pdu = PduEncHexOctet(pdu, atype ? 0x91 : 0x81);
    /* Destination address. */
    pdu = PduEncSemiOctetString(pdu, msg->msg_ms_addr + atype);
    /* Protocol identifier. */
    pdu = PduEncHexOctet(pdu, 0);
    /* Data coding scheme. */
    pdu = PduEncHexOctet(pdu, 0);
    /* Data length. */
    pdu = PduEncHexOctet(pdu, msg->msg_len);
    /* Data. */
    pdu = PduEncData(msg->msg_data, msg->msg_len, pdu, 0);

    *pdu = '\0';

    return pdubuf;
}

#if 0
#include <stdio.h>

void PduMsgDump(SHORT_MESSAGE *msg)
{
    char *cp;

    printf("-> %s", msg->msg_ms_addr);
    if (msg->msg_time) {
        struct _tm *ltm = localtime(&msg->msg_time);
        printf(" %02d.%02d.%04d", ltm->tm_mday, ltm->tm_mon + 1, ltm->tm_year + 1900);
        printf(" %02d:%02d:%02d", ltm->tm_hour, ltm->tm_min, ltm->tm_sec);
    }
    printf("\n   ");
    for (cp = (char *) msg->msg_data; *cp; cp++) {
        if (*cp >= 32 && *cp < 127) {
            putchar(*cp);
        } else {
            printf("[%02X]", *cp);
        }
    }
    putchar('\n');
}
#endif