/*
 * Copyright (C) 2013 by egnite GmbH
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

#include "serverwatch.h"

#include <sys/event.h>
#include <netinet/icmp.h>
#include <netinet/ip_icmp.h>
#include <netinet/in.h>

#include <stdio.h>

#include "ping.h"

static int IcmpCallback(NUTDEVICE * dev, NETBUF * nb)
{
    ICMPHDR *icp;
    uint_fast8_t i;

    /* ICMP header is in the transport part. */
    icp = (ICMPHDR *) nb->nb_tp.vp;

    if (icp && nb->nb_tp.sz >= sizeof(ICMPHDR)) {
        if (icp->icmp_type == ICMP_ECHOREPLY) {
            IPHDR *ip = (IPHDR *) nb->nb_nw.vp;

            for (i = 0; i < MAX_NODES && nodes[i].n_ip; i++) {
                if (ip->ip_src == nodes[i].n_ip) {
                    NutNetBufFree(nb);
                    NutEventBroadcast(&nodes[i].n_queue);

                    return 0;
                }
            }
        }
    }
    /* Pass packet back to the caller. */
    return -1;
}

static int SendPing(uint32_t dest, uint16_t id, uint16_t seq, int len)
{
    static uint_fast8_t initialized;
    NETBUF *nb;
    int i;
    uint8_t *dp;
    uint32_t spec;

    /* Register the ICMP callback when called for the first time. */
    if (!initialized) {
        if (NutRegisterIpHandler(IPPROTO_ICMP, IcmpCallback)) {
            return -1;
        }
        initialized = 1;
    }

    /* Create a new NETBUF for our ICMP telegram. */
    nb = NutNetBufAlloc(NULL, NBAF_APPLICATION, len);
    if (nb) {
        /* Fill the data area with sample characters. */
        dp = (uint8_t *) nb->nb_ap.vp;
        for (i = 0; i < len; i++) {
            *dp++ = 'a' + (i & 15);
        }
        /* Set up the echo request ID and sequence number. */
        spec = id;
        spec <<= 16;
        spec |= seq;
        /* Send out the packet. The name of this function is misleading. */
        if (NutIcmpReply(ICMP_ECHO, 0, htonl(spec), dest, nb) == 0) {
            /* Funny Nut/Net expects us to release the packet only if
               it has been successfully processed. */
            NutNetBufFree(nb);
            return 0;
        }
    }
    /* Return an error. */
    return -1;
}

/*!
 * \brief Ping a node and wait for response.
 *
 * \note This simple implementation needs modifications if used in another
 * application, because the ICMP callback accesses a global structure.
 *
 * \param dest IP address of the host to poll.
 * \param que  Nut/OS queue used by the ICMP callback.
 * \param tmo  Timeout in milliseconds.
 * \param retries Number of retries in case of timeouts.
 *
 * \return 0 on success, -1 if the destination node didn't respond.
 */
int Ping(uint32_t dest, HANDLE *que, uint32_t tmo, int_fast8_t retries)
{
    static uint16_t seq;

    do {
        /* Increment our sequence number. Actually we do not validate
           this in the response telegram. */
        seq++;
        /* Send an ICMP request and wait for response. */
        SendPing(dest, 1, seq, 32);
        if (NutEventWait(que, tmo) == 0) {
            /* Success, we got a response. */
            return 0;
        }
    } while (retries--);
    return -1;
}
