/*
 * Copyright (C) 2013 by egnite GmbH
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

#include "serverwatch.h"

#include <arch/avr.h>

#include <dev/debug.h>
#include <dev/i2cbus_avr.h>
#include <dev/usartavr.h>
#include <dev/lanc111.h>

#include <sys/version.h>
#include <sys/confnet.h>
#include <sys/event.h>
#include <sys/timer.h>
#include <sys/heap.h>
#include <sys/thread.h>
#include <sys/socket.h>

#include <netdb.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/ip_icmp.h>
#include <netinet/icmp.h>
#include <pro/dhcp.h>
#include <pro/snmp_agent.h>

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <time.h>

#include "locmote.h"
#include "simcom.h"
#include "ping.h"
#include "tcpchk.h"
#include "snmptrap.h"
#include "smsalert.h"
#include "thingspeak_update.h"
#include "xively_update.h"

/*!
 * \file serverwatch.c
 * \brief Main file of the Server Watch application.
 *
 * Server Watch polls several hard-coded hosts (including up to 7 TCP
 * ports) and sends SMS alerts if any of them is not reachable. In
 * addition, SNMP traps are sent on any status change and the current
 * status is reported to a data cloud, either Xively or Thingspeak.
 *
 * There is currently no user interface, all configuration is hard
 * coded.
 *
 * This example works on Ethernut 2 with a Locmote add-on board
 * attached.
 *
 * \verbatim
 * $Id$
 * \endverbatim
 */


#ifndef DEV_ETHER_NAME
#define DEV_ETHER_NAME  "eth0"
#endif

/*!
 * \brief Global structure containing configuration and status of polled nodes.
 */
NODE_INFO nodes[MAX_NODES];

/*!
 * \brief Configure the nodes.
 *
 * This sample configuration assumes, that we have a virtualization
 * server in the Internet, hosting two machines. One machine runs a
 * web server while the other is used by a mail server.
 */
static void ConfigNodes(void)
{
    int i;

    for (i = 0; i < MAX_NODES; i++) {
        nodes[i].n_icmp_timeout = 2000;
        nodes[i].n_icmp_retries = 3;
        nodes[i].n_up_interval = 60000;
        nodes[i].n_dn_interval = 10000;
    }

    nodes[0].n_name = "Internet router";
    nodes[0].n_ip = inet_addr("192.168.1.1");

    nodes[1].n_name = "Virtual server";
    nodes[1].n_ip = inet_addr("55.55.55.55");
    nodes[1].n_depnode = &nodes[0];

    nodes[2].n_name = "www.example.com";
    nodes[2].n_depnode = &nodes[1];
    nodes[2].n_ports[0] = 80;

    nodes[3].n_name = "mail.example.com";
    nodes[3].n_depnode = &nodes[1];
    nodes[3].n_ports[0] = 25;
    nodes[3].n_ports[1] = 110;
    nodes[3].n_ports[2] = 143;
}

/*!
 * \brief Print out a message and stop the device on fatal errors.
 *
 * A hardware reset is required to re-activate the device.
 */
static void Fatal(const char *msg)
{
    if (msg == NULL) {
        msg = "failed";
    }
    puts(msg);
    for (;;);
}

/*!
 * \brief Node polling thread.
 *
 * One thread is started for every monitored node.
 */
THREAD(ServerWatch, arg)
{
    NODE_INFO *node = arg;
    NODE_INFO *dep;
    uint_fast8_t down = 0;

    for (;;) {
        /* Monitor node if we got its IP address. */
        if (node->n_ip) {
            char *state;
            int subtype;

            /* Make cycle time depending on our current state. */
            if (node->n_last_down) {
                NutSleep(node->n_dn_interval);
            } else {
                NutSleep(node->n_up_interval);
            }

            /* Recursively check dependencies. If any node we depend on is
               down, then we skip our polling. */
            printf("Poll %s %s...", node->n_name, inet_ntoa(node->n_ip));
            for (dep = node->n_depnode; dep; dep = dep->n_depnode) {
                if (dep->n_last_down & 0x80) {
                    break;
                }
            }
            if (dep) {
                node->n_last_down = 0;
                puts("ignored");
                continue;
            }

            /* Try to get a ping response. */
            if (Ping(node->n_ip, &node->n_queue, 2000, node->n_icmp_retries)) {
                down = 0x80;
                state = "down";
                subtype = 2;
            } else {
                down &= ~0x80;
                state = "up";
                subtype = 1;
            }
            puts(state);

            /* Immediately send a trap on any node status change. */
            if ((down ^ node->n_last_down) & 0x80) {
#if defined(TRAP_TARGET)
                char *msg;

                asprintf(&msg, "%s %s %s", node->n_name, inet_ntoa(node->n_ip), state);
                SendSnmpTrap(subtype, inet_addr(TRAP_TARGET), node->n_ip, msg);
                free(msg);
#endif
            }

            /* If the node is up, check its configured TCP ports. */
            else if (down != 0x80) {
                int pidx;

                for (pidx = 0; pidx < MAX_PORTS && node->n_ports[pidx]; pidx++) {
                    NutSleep(2000);
                    printf("Connect %s:%u...", node->n_name, node->n_ports[pidx]);
                    if (TcpPortPoll(node->n_ip, node->n_ports[pidx])) {
                        down |= 1 << pidx;
                        state = "down";
                        subtype = 4;
                    } else {
                        down &= ~(1 << pidx);
                        state = "up";
                        subtype = 3;
                    }
                    puts(state);
                }
                for (pidx = 0; pidx < MAX_PORTS && node->n_ports[pidx]; pidx++) {
                    /* Immediately send a trap on any port status change. */
                    if ((down ^ node->n_last_down) & (1 << pidx)) {
#if defined(TRAP_TARGET)
                        char *msg;

                        asprintf(&msg, "%s %s:%u %s", node->n_name, inet_ntoa(node->n_ip), node->n_ports[pidx], state);
                        SendSnmpTrap(subtype, inet_addr(TRAP_TARGET), node->n_ip, msg);
                        free(msg);
#endif
                    }
                }
            }
            node->n_last_down = down;
        }
        /* If the IP address is unknown, try DNS, using the configured node name. */
        else {
            printf("Query IP of %s...", node->n_name);
            node->n_ip = NutDnsGetHostByName((uint8_t*) node->n_name);
            if (node->n_ip) {
                puts(inet_ntoa(node->n_ip));
            } else {
                puts("failed");
            }
        }
    }
}

/*!
 * \brief Initialize the modem's UART interface.
 *
 * See serverwatch.h for parameters.
 *
 * \return Pointer to the associated stream.
 */
static FILE *InitModemUart(void)
{
    FILE *stream = NULL;

    if (NutRegisterDevice(&MODEM_UARTDEV, 0, 0) == 0) {
        stream = fopen(MODEM_UARTDEV.dev_name, "r+b");
        if (stream) {
            uint32_t parm = MODEM_BAUDRATE;
            _ioctl(_fileno(stream), UART_SETSPEED, &parm);
            parm = MODEM_BUFFSIZE;
            _ioctl(_fileno(stream), UART_SETRXBUFSIZ, &parm);
        }
    }
    return stream;
}

static void InitLocmote(void)
{
    uint32_t lctl;

    printf("Init I2C...");
    /* Enable I2C pull-ups. */
    sbi(PORTD, 0);
    sbi(PORTD, 1);

    if (NutRegisterI2cSlave(&i2cLocmote, &i2cBus0Avr)) {
        Fatal(NULL);
    }
    printf("speed...");
    lctl = NutI2cBusRate(&i2cBus0Avr, 20000);
    lctl = NutI2cBusRate(&i2cBus0Avr, 20000);
    printf("%lu bps\n", lctl);

    for (;;) {
        NutSleep(1000);
        printf("\nLocmote power on ...");
        if (LocmotePower(1)) {
            puts(" failed");
            LocmotePower(0);
            NutSleep(1000);
        } else {
            puts(" OK");
            break;
        }
    }
    /* Switch to GSM channel. */
    LocmoteSetBit(LMC_GPS_BIT, 0);
}

static void InitModem(FILE *stream)
{
    SimFlush(stream, 1000);
    SimSwitchTimeout(stream, 2000);
    while (SimCmdOk(stream, "AT", 3)) {
        LocmotePower(0);
        NutSleep(5000);
        LocmotePower(1);
    }
    SimSwitchTimeout(stream, 200);
    SimCmdOk(stream, "ATZ", 3);
    SimCmdOk(stream, "ATE0", 3);
    SimFlush(stream, 5000);
    SimCmdOk(stream, "AT+CSCA?", 3);

    SimMsgTextMode(stream, 0);
}

static void DataCloudUpdate(void)
{
#if defined(XIVELY_FEED_ID) && defined(XIVELY_API_KEY)
    XivelyUpdate(XIVELY_FEED_ID, XIVELY_API_KEY);
#endif
#if defined(THINGSPEAK_API_KEY)
    ThingSpeakUpdate(THINGSPEAK_API_KEY);
#endif
}

static void AddNodeAlert(SMS_ALERT *smsa, int nidx)
{
    char *line;

    asprintf(&line, SMS_REPORT_NODE, nodes[nidx].n_name);
    SmsAlertWriteString(smsa, line);
    free(line);
}

static void AddPortAlert(SMS_ALERT *smsa, int nidx, int pidx)
{
    if (smsa) {
        char *line;

        asprintf(&line, SMS_REPORT_PORT, nodes[nidx].n_name, nodes[nidx].n_ports[pidx]);
        SmsAlertWriteString(smsa, line);
        free(line);
    }
}

/*!
 * \brief Alarm reporting loop.
 *
 * This function never returns.
 *
 * \param modem Pointer to a stream that is associated to the modem
 *              interface.
 */
static void AlarmReporter(FILE *modem) NUT_NORETURN_FUNC;
void AlarmReporter(FILE *modem)
{
    static uint8_t reported[MAX_NODES];
    static uint8_t xi_reported[MAX_NODES];
    static uint8_t nodes_down;
    static uint8_t ports_down;
    static uint8_t nodes_up;
    static uint8_t ports_up;
    static uint8_t count[MAX_NODES][8];
    int nidx;
    int pidx;
    int slowscan = 0;
    SMS_ALERT *smsa;
    int seconds = 0;

    /* Initial data cloud update on system start. */
    DataCloudUpdate();
    memcpy(xi_reported, reported, sizeof(xi_reported));
    for (;;) {
        /*
         * Loop delay.
         *
         * Usually the status is scanned every second. However, if we
         * reported a down event, then we slow down scanning to one
         * minute. This should help to collect more events into
         * a single SMS.
         */
        if (slowscan) {
            NutSleep(60000);
            seconds += 60;
        } else {
            NutSleep(1000);
            seconds++;
        }

        /*
         * Update the cloud data every 10 minutes.
         */
        if (seconds > 600 || memcmp(xi_reported, reported, sizeof(xi_reported))) {
            DataCloudUpdate();
            memcpy(xi_reported, reported, sizeof(xi_reported));
            seconds = 0;
        }

        /*
         * Count the number of unreported down/up events.
         */
        nodes_down = 0;
        ports_down = 0;
        nodes_up = 0;
        ports_up = 0;
        for (nidx = 0; nidx < MAX_NODES; nidx++) {
            for (pidx = 0; pidx < 8; pidx++) {
                if (nodes[nidx].n_last_down & (1 << pidx)) {
                    if (count[nidx][pidx] < 120) {
                        count[nidx][pidx]++;
                    }
                    else if ((reported[nidx] & (1 << pidx)) == 0) {
                        if (pidx == 7) {
                            nodes_down++;
                        } else {
                            ports_down++;
                        }
                    }
                } else {
                    count[nidx][pidx] = 0;
                    if (reported[nidx] & (1 << pidx)) {
                        if (pidx == 7) {
                            nodes_up++;
                        } else {
                            ports_up++;
                        }
                    }
                }
            }
        }

        /*
         * Send SMS alerts.
         *
         * Note, that only one type of reports (node up/down or port up/down)
         * is sent on each loop. The underlying routines will take care, that
         * the report is divided into several messages, if the list won't fit
         * in a single SMS.
         */
        slowscan = 0;
        smsa = NULL;
        if (nodes_down) {
            /* Report nodes down. */
            smsa = SmsAlertOpen(modem, SMS_TARGET, SMS_SUBJECT_NODES_DOWN);
            for (nidx = 0; nidx < MAX_NODES && nodes[nidx].n_name; nidx++) {
                if ((reported[nidx] & 0x80) == 0 && (nodes[nidx].n_last_down & 0x80) != 0) {
                    AddNodeAlert(smsa, nidx);
                    reported[nidx] |= 0x80;
                    slowscan = 1;
                }
            }
        }
        else if (ports_down) {
            /* Report ports down. */
            smsa = SmsAlertOpen(modem, SMS_TARGET, SMS_SUBJECT_PORTS_DOWN);
            for (nidx = 0; nidx < MAX_NODES && nodes[nidx].n_name; nidx++) {
                for (pidx = 0; pidx < MAX_PORTS && nodes[nidx].n_ports[pidx]; pidx++) {
                    uint_fast8_t msk = 1 << pidx;
                    if ((reported[nidx] & msk) == 0 && (nodes[nidx].n_last_down & msk) != 0) {
                        AddPortAlert(smsa, nidx, pidx);
                        reported[nidx] |= msk;
                        slowscan = 1;
                    }
                }
            }
        }
        else if (nodes_up) {
            /* Report nodes up. */
            smsa = SmsAlertOpen(modem, SMS_TARGET, SMS_SUBJECT_NODES_UP);
            for (nidx = 0; nidx < MAX_NODES && nodes[nidx].n_name; nidx++) {
                if ((reported[nidx] & 0x80) != 0 && (nodes[nidx].n_last_down & 0x80) == 0) {
                    AddNodeAlert(smsa, nidx);
                    reported[nidx] &= ~0x80;
                }
            }
        }
        else if (ports_up) {
            /* Report ports up. */
            smsa = SmsAlertOpen(modem, SMS_TARGET, SMS_SUBJECT_PORTS_UP);
            for (nidx = 0; nidx < MAX_NODES && nodes[nidx].n_name; nidx++) {
                for (pidx = 0; pidx < MAX_PORTS && nodes[nidx].n_ports[pidx]; pidx++) {
                    uint_fast8_t msk = 1 << pidx;
                    if ((reported[nidx] & msk) != 0 && (nodes[nidx].n_last_down & msk) == 0) {
                        AddPortAlert(smsa, nidx, pidx);
                        reported[nidx] &= ~msk;
                    }
                }
            }
        }
        if (smsa) {
            SmsAlertClose(smsa);
        }
    }
}

int main(void)
{
    int nidx;
    FILE *uart;

    /* Enable Locmote early. */
    LocmoteEnable(1);

    NutRegisterDevice(&devDebug0, 0, 0);
    freopen("uart0", "w", stdout);
    printf("\n\nServerWatch - Nut/OS %s - " __DATE__ " " __TIME__ "\n", NutVersionString());

    InitLocmote();

    printf("Register Ethernet %s...", DEV_ETHER.dev_name);
    if (NutRegisterDevice(&DEV_ETHER, 0, 0)) {
        Fatal(NULL);
    }
    puts("OK");

    printf("Configure Ethernet...");
    if (NutDhcpIfConfig(DEV_ETHER_NAME, 0, 60000)) {
        Fatal(NULL);
    }
    printf("%s ready\n", inet_ntoa(confnet.cdn_ip_addr));

    ConfigNodes();

    printf("Initialize Modem UART...");
    uart = InitModemUart();
    if (uart == NULL) {
        Fatal(NULL);
    }
    puts("OK");

    InitModem(uart);

    for (nidx = 0; nidx < 5; nidx++) {
        NutThreadCreate("watcher", ServerWatch, &nodes[nidx], 1024);
        NutSleep(5000);
    }

#if 0
    /* Optionally send SMS on system start. You may also enable this part
       to test SMS delivery. */
    printf("Send system start SMS...");
    if (SimMsgSend(uart, SMS_TARGET, "ServerWatch started")) {
        puts("failed");
    } else {
        puts("OK");
    }
#endif

    /* Enter alarm reporting loop. Will never return. */
    AlarmReporter(uart);

    return 0;
}
