/*
 * Copyright (C) 2013 by egnite GmbH
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

#include <dev/uart.h>
#include <sys/timer.h>
#include <sys/syslog.h>

#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <time.h>

#include "simcom.h"

static char rspbuf[512];

/*!
 * \brief Change timeout.
 *
 * \param modem Associated modem communication stream.
 * \param tmo   New timeout in milliseconds.
 *
 * \return Previous timeout value.
 */
uint32_t SimSwitchTimeout(FILE *modem, uint32_t tmo)
{
    uint32_t ret;

    _ioctl(_fileno(modem), UART_GETREADTIMEOUT, &ret);
    if (ret != tmo) {
        _ioctl(_fileno(modem), UART_SETREADTIMEOUT, &tmo);
        printf("TMO: %lu -> %lu\n", ret, tmo);
    }
    return ret;
}

/*!
 * \brief Flush input buffer.
 *
 * \param modem Associated modem communication stream.
 * \param tmo   Timeout in milliseconds.
 */
void SimFlush(FILE *modem, uint32_t tmo)
{
    int ch;

    fflush(modem);
    NutSleep(200);
    tmo = SimSwitchTimeout(modem, tmo);
    printf("FLU: ");
    for (;;) {
        ch = fgetc(modem);
        if (ch == EOF) {
            break;
        }
        if (ch >= 32 && ch <= 126) {
            putchar(ch);
        } else {
            printf("[%02x]", ch);
        }
    }
    putchar('\n');
    SimSwitchTimeout(modem, tmo);
}

/*!
 * \brief Send command and get response.
 *
 * \param modem Associated modem communication stream.
 * \param cmd   Modem command to send.
 * \param rsp   Buffer that receives the response.
 * \param size  Size of the response buffer.
 *
 * \return Pointer to the response buffer.
 */
char *SimCmd(FILE *modem, char *cmd, char *rsp, int siz)
{
    if (cmd) {
        fputs(cmd, modem);
        fputc('\r', modem);
        fflush(modem);
        printf("CMD(%03d): %s\n", strlen(cmd), cmd);
        //NutSleep(500);
    }
    if (rsp && siz) {
        rsp[--siz] = '\0';
        for (;;) {
            char *cp;

            if (fgets(rsp, siz, modem) == NULL) {
                *rsp = '\0';
                break;
            }
            cp = strchr(rsp, '\r');
            if (cp == NULL) {
                cp = strchr(rsp, '\n');
            }
            if (cp) {
                *cp = '\0';
                if (*rsp) {
                    printf("RSP(%03d): %s\n", strlen(rsp), rsp);
                    break;
                }
            }
        }
    }
    return rsp;
}

/*!
 * \brief Send command and wait for OK response.
 *
 * \param modem   Associated modem communication stream.
 * \param cmd     Modem command to send.
 * \param retries Number of retries.
 */
int SimCmdOk(FILE *modem, char *cmd, int retries)
{
    static char rsp[32];

    do {
        SimCmd(modem, cmd, rsp, sizeof(rsp));
        while (rsp[0]) {
            if (strncmp(rsp, "OK", 2) == 0) {
                return 0;
            }
            SimCmd(modem, NULL, rsp, sizeof(rsp));
        }
        NutSleep(100);
    } while (retries-- > 0);

    return -1;
}

/*!
 * \brief Set text format for SMS.
 *
 * \param modem Associated modem communication stream.
 * \param on    Set to 0 to disable text mode. Any other value will enable it.
 *
 * \return 0 on success, -1 otherwise.
 */
int SimMsgTextMode(FILE *modem, int on)
{
    int rc;
    char *cmd;

    asprintf(&cmd, "AT+CMGF=%d", on != 0);
    rc = SimCmdOk(modem, cmd, 3);
    free(cmd);

    return rc;
}

/*!
 * \brief Load SMS directory.
 *
 * \param modem Associated modem communication stream.
 */
SMS_DIRECTORY *SimMsgListLoad(FILE *modem, int state)
{
    char *cmd;
    uint32_t tmo;
    SMS_DIRECTORY *dir = NULL;

    asprintf(&cmd, "AT+CMGL=%d,%d", state & SMS_STATE_MASK, (state & SMS_STATE_KEEP) != 0);
    SimCmd(modem, cmd, NULL, 0);
    free(cmd);

    tmo = SimSwitchTimeout(modem, 5000);
    NutSleep(500);
    while (1) {
        SimCmd(modem, NULL, rspbuf, sizeof(rspbuf));
        if (rspbuf[0] == '\0') {
            break;
        }
        if (strncmp(rspbuf, "OK", 2) == 0) {
            break;
        }
        else if (strncmp(rspbuf, "+CMGL:", 6) == 0) {
            if (dir == NULL) {
                dir = (SMS_DIRECTORY *) calloc(1, sizeof(SMS_DIRECTORY));
            }
            if (dir) {
                char *cp = rspbuf + 6;
                dir->dir_ent[dir->dir_len].ent_index = atoi(rspbuf + 6);
                cp = strchr(cp, ',');
                if (cp) {
                    dir->dir_ent[dir->dir_len].ent_status = atoi(cp + 1);
                    SimCmd(modem, NULL, rspbuf, sizeof(rspbuf));
                    dir->dir_ent[dir->dir_len].ent_msg = PduDecMsg(rspbuf, 0);
                    dir->dir_len++;
                }
            }
        }
    }
    SimSwitchTimeout(modem, tmo);

    return dir;
}

/*!
 * \brief Release SMS directory.
 *
 * \param dir Pointer to an SMS directory, obtained by a previous call
 *            to SimMsgListLoad().
 */
void SimMsgListUnload(SMS_DIRECTORY *dir)
{
    int i;

    for (i = 0; i < dir->dir_len; i++) {
        //free(dir->dir_entry[i]);
        free(dir->dir_ent[i].ent_msg->msg_data);
        free(dir->dir_ent[i].ent_msg);
    }
    free(dir);
}

/*!
 * \brief Delete SMS.
 *
 * \param modem Associated modem communication stream.
 * \param idx   Index to delete.
 *
 * \return 0 on success, -1 otherwise.
 */
int SimMsgDelIndex(FILE *modem, int idx)
{
    int rc;
    char *cmd;

    asprintf(&cmd, "AT+CMGD=%d", idx);
    rc = SimCmdOk(modem, cmd, 3);
    free(cmd);

    return rc;
}

/*!
 * \brief Send SMS reply.
 *
 * \param modem Associated modem communication stream.
 * \param data  Message data to send.
 *
 * \return 0 on success, -1 otherwise.
 */
int SimMsgRespond(FILE *modem, SHORT_MESSAGE *msg, const char *data)
{
    int rc = 0;
    char *cmd;
    SHORT_MESSAGE *rmsg;
    SHORT_MESSAGE *tmsg;
    char *pdu;

    SimFlush(modem, 2000);

    rmsg = (SHORT_MESSAGE *) calloc(1, sizeof(SHORT_MESSAGE));
    rmsg->msg_flags = TP_MTI_SUBMIT << TP_MTI_LSB;
    strcpy(rmsg->msg_ms_addr, msg->msg_ms_addr);
    rmsg->msg_len = strlen(data);
    rmsg->msg_data = (uint8_t *) strdup(data);
    pdu = PduEncMsg(rmsg);

    tmsg = PduDecMsg(pdu, 1);
    free(tmsg->msg_data);
    free(tmsg);

#if 1
    asprintf(&cmd, "AT+CMGS=%d", (strlen(pdu) / 2) - 1);
    SimCmd(modem, cmd, NULL, 0);
    free(cmd);
    SimFlush(modem, 2000);
    fputs(pdu, modem);
    fputc('\x1A', modem);
    fflush(modem);
    printf("PDU(%03d): %s\n", strlen(pdu), pdu);
    rc = SimCmdOk(modem, NULL, 64);
    if (rc) {
        syslog(LOG_INFO, "SMS response to %s failed", msg->msg_ms_addr);
    } else {
        syslog(LOG_INFO, "SMS to %s: %s", msg->msg_ms_addr, data);
    }
#endif

    free(pdu);
    free(rmsg->msg_data);
    free(rmsg);

    return rc;
}

/*!
 * \brief Send SMS.
 *
 * \param modem Associated modem communication stream.
 * \param addr  Destination phone number.
 * \param data  Message data to send.
 *
 * \return 0 on success, -1 otherwise.
 */
int SimMsgSend(FILE *modem, const char *addr, const char *data)
{
    int rc = 0;
    char *cmd;
    SHORT_MESSAGE *msg;
    char *pdu;

    SimFlush(modem, 2000);

    msg = (SHORT_MESSAGE *) calloc(1, sizeof(SHORT_MESSAGE));
    msg->msg_flags = TP_MTI_SUBMIT << TP_MTI_LSB;
    strcpy(msg->msg_ms_addr, addr);
    msg->msg_len = strlen(data);
    msg->msg_data = (uint8_t *) strdup(data);
    pdu = PduEncMsg(msg);

    asprintf(&cmd, "AT+CMGS=%d", (strlen(pdu) / 2) - 1);
    SimCmd(modem, cmd, NULL, 0);
    free(cmd);
    SimFlush(modem, 2000);
    fputs(pdu, modem);
    fputc('\x1A', modem);
    fflush(modem);
    printf("PDU(%03d): %s\n", strlen(pdu), pdu);
    rc = SimCmdOk(modem, NULL, 64);
    rc = SimCmdOk(modem, NULL, 64);

    free(pdu);
    free(msg->msg_data);
    free(msg);

    return rc;
}
