/*!
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file config.c
 * \brief Radio configuration.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <cfg/os.h>
#include <cfg/clock.h>
#include <dev/board.h>
#include <dev/at45db.h>
#include <dev/at91_spi.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>

#include <sys/version.h>
#include <sys/confnet.h>
#include <sys/atom.h>
#include <sys/heap.h>
#include <sys/thread.h>
#include <sys/timer.h>
#include <sys/event.h>
#include <sys/socket.h>

#if defined(USE_SOFTWARE_CODEC)
#include <hxmp3/mp3dec.h>
#endif

#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <pro/dhcp.h>

#include "config.h"
#include "logmsg.h"
#include "favlist.h"
#include "webradio.h"

#define FAVLIST_MAGIC   "FL1"
#define FAVLIST_SECTOR  16


RADIOSTATION favlist[MAX_FAVORITES];

/*!
 * \brief Clear favorites list.
 *
 * Will be called by FavListLoad() before loading a new list.
 */
static void FavListClear(void)
{
    int idx;
    int i;

    /* Release all memory occupied by the current configuration. */
    for (idx = 0; idx < MAX_FAVORITES; idx++) {
        if (favlist[idx].rs_name) {
            free(favlist[idx].rs_name);
        }
        for (i = 0; i < favlist[idx].rs_streams; i++) {
            if (favlist[idx].rs_uri[i]) {
                free(favlist[idx].rs_uri[i]);
            }
        }
    }
    memset(favlist, 0, MAX_FAVORITES * sizeof(RADIOSTATION));
}

/*!
 * \brief Set favorite list entry.
 *
 * \param idx  Index of the entry. Use MAX_FAVORITES to add a new entry.
 * \param name Name of the station. Must be provided, if a new station
 *             is added. May be NULL when adding a URI. When used with
 *             an existing entry, the current name will be replaced.
 * \param uri  URI of the stream, that will be added. If NULL, then all
 *             existing entries are cleared. If both, the name and the
 *             URI are NULL, then the list entry is cleared.
 *
 * \return Entry index on success, -1 otherwise.
 */
int FavListSet(int idx, CONST char *name, CONST char * uri)
{
    int i;

    LogMsg(LOG_CONFIG, "Set %d,%s,%s\n", idx, name, uri);
    if (idx >= MAX_FAVORITES) {
        /* New entry. Find a free slot. */
        idx = LAST_FAVORITE + 1;
        while (idx < MAX_FAVORITES && favlist[idx].rs_name) {
            idx++;
        }
    }
    if (idx >= MAX_FAVORITES) {
        /* We are full. */
        return -1;
    }
    if (name || (name == NULL && uri == NULL)) {
        if (favlist[idx].rs_name) {
            free(favlist[idx].rs_name);
        }
        favlist[idx].rs_name = NULL;
    }
    if (name) {
        favlist[idx].rs_name = strdup(name);
    }

    if (uri) {
        if (favlist[idx].rs_streams < MAXNUM_STREAMS) {
            favlist[idx].rs_uri[favlist[idx].rs_streams] = strdup(uri);
            favlist[idx].rs_streams++;
        }
        else {
            idx = -1;
        }
    }
    else {
        for (i = 0; i < favlist[idx].rs_streams; i++) {
            if (favlist[idx].rs_uri[i]) {
                free(favlist[idx].rs_uri[i]);
            }
            favlist[idx].rs_uri[i] = NULL;
        }
        favlist[idx].rs_streams = 0;
    }
    return idx;
}

/*!
 * \brief Copy a favorites list entry.
 *
 * \param src Index of the source entry.
 * \param dst Index of the destination entry.
 *
 * \return 0 on success, -1 otherwise.
 */
int FavListCopy(int src, int dst)
{
    int i;

    /*
     * Release an existing entry at the destination index.
     */
    LogMsg(LOG_CONFIG, "Copying %d to %d\n", src, dst);
    if (favlist[dst].rs_name) {
        free(favlist[dst].rs_name);
    }
    for (i = 0; i < favlist[dst].rs_streams; i++) {
        if (favlist[dst].rs_uri[i]) {
            free(favlist[dst].rs_uri[i]);
        }
    }
    memset(&favlist[dst], 0, sizeof(RADIOSTATION));

    /*
     * Copy the source entry to the destination index.
     */
    if (favlist[src].rs_name) {
        favlist[dst].rs_name = strdup(favlist[src].rs_name);
        for (i = 0; i < favlist[src].rs_streams; i++) {
            if (favlist[src].rs_uri[i]) {
                favlist[dst].rs_uri[favlist[dst].rs_streams++] = strdup(favlist[src].rs_uri[i]);
            }
        }
        return 0;
    }
    return -1;
}

/*!
 * \brief Search favorite radio station.
 *
 * \param idx Start index.
 * \param dir Direction to search, 1 (forward) or -1 (backwards).
 *
 * \return Index of the new station or -1 if none available.
 */
int FavListSearch(int idx, int dir)
{
    int rc = idx;

    LogMsg(LOG_CONFIG, "Searching %d %d\n", idx, dir);
    for (;;) {
        rc += dir;
        if (rc <= LAST_FAVORITE) {
            rc = MAX_FAVORITES - 1;
        }
        else if (rc >= MAX_FAVORITES) {
            rc = LAST_FAVORITE + 1;
        }
        if ((favlist[rc].rs_name && favlist[rc].rs_streams)) {
            /* New entry found. */
            break;
        }
        if (rc == idx) {
            /* Whole list searched, but no entry found. */
            return -1;
        }
    }
    return rc;
}

/*!
 * \brief Calculate total size of the favorites list.
 *
 * \return Number of bytes used.
 */
size_t FavListSize(void)
{
    size_t rc = 0;
    int idx;
    int i;
    RADIOSTATION *flp;

    for (idx = 1; idx < MAX_FAVORITES; idx++) {
        flp = &favlist[idx];
        if (flp->rs_name) {
            rc += strlen(flp->rs_name) + 1;
            for (i = 0; i < flp->rs_streams; i++) {
                if (flp->rs_uri[i]) {
                    rc += strlen(flp->rs_uri[i]) + 1;
                }
            }
        }
    }
    return rc + 1;
}

/*!
 * \brief Save favorites list in non-volatile memory.
 *
 * \return 0 on success, -1 otherwise.
 */
int FavListSave(void)
{
    int idx;
    int i;
    RADIOSTATION *rsp;

    /* Rewind and save the magic name. */
    ConfigRewind(FAVLIST_SECTOR);
    ConfigSaveString(FAVLIST_MAGIC);

    /* Save stations. */
    for (idx = LAST_FAVORITE; idx < MAX_FAVORITES; idx++) {
        rsp = &favlist[idx];
        if (rsp->rs_name) {
            LogMsg(LOG_CONFIG, "Saving %d %s\n", idx, rsp->rs_name);
            ConfigSaveString(rsp->rs_name);
            for (i = 0; i < rsp->rs_streams; i++) {
                if (rsp->rs_uri[i]) {
                    LogMsg(LOG_CONFIG, "  URI=%s\n", rsp->rs_uri[i]);
                    ConfigSaveString(rsp->rs_uri[i]);
                }
            }
            ConfigSaveString("");
        }
        else if (idx == LAST_FAVORITE) {
            ConfigSaveString("None");
            ConfigSaveString("");
        }
        NutThreadYield();
    }
    ConfigSaveString("");
    ConfigFlush();

    return 0;
}

/*!
 * \brief Load favorites list from non-volatile memory.
 *
 * \return 0 on success, -1 otherwise.
 */
int FavListLoad(void)
{
    int rc = -1;
    char *buf;
    int idx;
    int i;

    buf = malloc(255);
    ConfigRewind(FAVLIST_SECTOR);
    ConfigLoadString(buf, 4);
    if (strcmp(buf, FAVLIST_MAGIC) == 0) {
        FavListClear();
        rc = 0;

        for (idx = LAST_FAVORITE; idx < MAX_FAVORITES; idx++) {
            ConfigLoadString(buf, 255);
            if (*buf == 0) {
                break;
            }
            favlist[idx].rs_name = strdup(buf);
            LogMsg(LOG_CONFIG, "Loading %d %s\n", idx, favlist[idx].rs_name);
            for (i = 0; i < MAXNUM_STREAMS; i++) {
                ConfigLoadString(buf, 255);
                if (*buf == 0) {
                    break;
                }
                LogMsg(LOG_CONFIG, "  URI=%s\n", buf);
                favlist[idx].rs_uri[i] = strdup(buf);
                favlist[idx].rs_streams++;
            }
        }
    }
    free(buf);

    return rc;
}

/*!
 * \brief Set pre-configured favorites list.
 */
void FavListResetFactory(void)
{
    int idx;

    FavListClear();

    /* 
     * Add pre-configured radio stations. 
     */
    if ((idx = FavListSet(MAX_FAVORITES, "FREQUENCE3", "88.191.22.199:8000")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "194.158.114.68:8000");
    FavListSet(idx, NULL, "194.158.114.67:8000");
    FavListSet(idx, NULL, "193.251.154.243:8000");
    FavListSet(idx, NULL, "213.186.56.206:8000");
    FavListSet(idx, NULL, "194.158.114.66:8000");
    FavListSet(idx, NULL, "213.186.41.160:8000");
    FavListSet(idx, NULL, "88.191.11.19:8000");
    FavListSet(idx, NULL, "62.4.21.91:8000");
    FavListSet(idx, NULL, "212.129.62.140:8000");
    FavListSet(idx, NULL, "194.117.194.66:8000");
    FavListSet(idx, NULL, "213.186.37.124:8000");
    FavListSet(idx, NULL, "80.65.234.120:8000");
    FavListSet(idx, NULL, "213.161.201.21:8000");
    FavListSet(idx, NULL, "213.186.60.54:8000");
    FavListSet(idx, NULL, "193.251.154.242:8000");
    FavListSet(idx, NULL, "83.143.18.76:8000");
    FavListSet(idx, NULL, "193.17.192.35:8000");
    FavListSet(idx, NULL, "193.222.128.135:8000");

    /* Start with this reliable station. */
    FavListCopy(idx, LAST_FAVORITE);

    if ((idx = FavListSet(MAX_FAVORITES, "FREQUENCE3 Ogg", "ogg.frequence3.net:19000/frequence3.ogg")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "ganymede.frequence3.net:19000/frequence3.ogg");
    FavListSet(idx, NULL, "aspairt.frequence3.net:19000/frequence3.ogg");
    FavListSet(idx, NULL, "mao.frequence3.net:19000/frequence3.ogg");
    FavListSet(idx, NULL, "pluton.frequence3.net:19000/frequence3.ogg");

    if ((idx = FavListSet(MAX_FAVORITES, "VPR Classical", "205.234.168.42:8000")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "205.234.168.42:80");

    if (FavListSet(MAX_FAVORITES, "All Classical 89.9 KBPS Portland", "216.218.147.60:8220") == -1) {
        return;
    }

    if ((idx = FavListSet(MAX_FAVORITES, ".977 Hitz Channel", "64.236.34.97/stream/1074")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "205.188.215.230:8002");

    if ((idx = FavListSet(MAX_FAVORITES, "Rock'One", "88.191.12.222:6005")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "80.65.234.119:6005");
    FavListSet(idx, NULL, "194.117.194.66:6005");

    if (FavListSet(MAX_FAVORITES, "veNet Radio", "71.62.186.136:1039") == -1) {
        return;
    }
    if (FavListSet(MAX_FAVORITES, "gamer-FM", "82.149.224.55:8500") == -1) {
        return;
    }

    if ((idx = FavListSet(MAX_FAVORITES, "BigB", "70.87.68.38:9000")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "74.52.7.164:9000");
    FavListSet(idx, NULL, "67.19.249.226:8000");

    if ((idx = FavListSet(MAX_FAVORITES, "SKY.FM Hits", "212.23.5.99:8006")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "160.79.128.62:6628");
    FavListSet(idx, NULL, "64.74.207.40:8628");

    if ((idx = FavListSet(MAX_FAVORITES, "181.FM Classic Hits", "scfire-nyk-aa01.stream.aol.com:80/stream/1094")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "scfire-nyk-aa02.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-nyk-aa03.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-nyk-aa04.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-nyk-aa05.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-dll-aa02.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-dll-aa03.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-dll-aa04.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-dll-aa05.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-chi-aa01.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-chi-aa02.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-chi-aa03.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-chi-aa04.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-chi-aa05.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-ntc-aa01.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-ntc-aa02.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-ntc-aa03.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-ntc-aa04.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "scfire-ntc-aa05.stream.aol.com:80/stream/1094");
    FavListSet(idx, NULL, "207.200.96.227:8008");

    if ((idx = FavListSet(MAX_FAVORITES, "Virgin Radio", "ice-mp3-vr-128.smgradio.com:80/vr.mp3")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "85.159.184.44:80");

    if (FavListSet(MAX_FAVORITES, "Virgin Radio Ogg", "ogg2.smgradio.com:80/gr32.ogg") == -1) {
        return;
    }

    if (FavListSet(MAX_FAVORITES, "Metal Only", "80.67.23.74:6665") == -1) {
        return;
    }

    if ((idx = FavListSet(MAX_FAVORITES, "Radio K", "160.94.138.250:8128")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "radiokstreams.cce.umn.edu:8128");

    if ((idx = FavListSet(MAX_FAVORITES, "SpacialNet", "64.124.106.176:22122")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "sc9.spacialnet.com:22122");

    if ((idx = FavListSet(MAX_FAVORITES, "Blue Sugar Radio", "209.51.162.170:9560")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "66.92.79.161:8000");

    if (FavListSet(MAX_FAVORITES, "Pre Party Radio", "ds130.twilightinc.nl:8000/ogg-128-prepartyradio-com.ogg") == -1) {
        return;
    }

    if (FavListSet(MAX_FAVORITES, "KAMP Student Radio", "128.196.31.10:8000/high.ogg") == -1) {
        return;
    }

    if (FavListSet(MAX_FAVORITES, "Project FX", "85.25.134.48:8000/evil.ogg") == -1) {
        return;
    }
    if (FavListSet(MAX_FAVORITES, "Coco Radio", "67.159.26.47:1010/Cocoradio32k.ogg?refid=1") == -1) {
        return;
    }
    if (FavListSet(MAX_FAVORITES, "European Music Radio", "nednl.net:8000/emr.ogg") == -1) {
        return;
    }
    if (FavListSet(MAX_FAVORITES, "Radio Bangkok", "216.66.69.100:8000/live.ogg") == -1) {
        return;
    }
    if (FavListSet(MAX_FAVORITES, "Radio Swiss Classic", "stream-1.ssatr.ch:80/rsc32.ogg") == -1) {
        return;
    }
    if ((idx = FavListSet(MAX_FAVORITES, "Eins Live", "gffstream.ic.llnwd.net/stream/gffstream_stream_wdr_einslive_a")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_stream_wdr_einslive_b");

    if ((idx = FavListSet(MAX_FAVORITES, "Eins Live Diggi", "gffstream.ic.llnwd.net/stream/gffstream_einslive_a")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_einslive_b");

    if ((idx = FavListSet(MAX_FAVORITES, "Eins Live Kunst", "gffstream.ic.llnwd.net/stream/gffstream_gffstream5")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_gffstream6");

    if ((idx = FavListSet(MAX_FAVORITES, "WDR 2", "gffstream.ic.llnwd.net/stream/gffstream_w17b")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_w17a");

    if ((idx = FavListSet(MAX_FAVORITES, "WDR 3", "gffstream.ic.llnwd.net/stream/gffstream_w21a")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_w21b");

    if ((idx = FavListSet(MAX_FAVORITES, "WDR 4", "gffstream.ic.llnwd.net/stream/gffstream_w18b")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_w18a");

    if ((idx = FavListSet(MAX_FAVORITES, "WDR 5", "gffstream.ic.llnwd.net/stream/gffstream_w19a")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_w19b");

    if ((idx = FavListSet(MAX_FAVORITES, "Kiraka", "gffstream.ic.llnwd.net/stream/gffstream_gffwdr1a")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "gffstream.ic.llnwd.net/stream/gffstream_gffwdr1b");


    if ((idx = FavListSet(MAX_FAVORITES, "Decades Radio", "203.16.26.101:80/decades192.ogg")) == -1) {
        return;
    }
    FavListSet(idx, NULL, "bcast1.qld.abbn.tv:80/decades192.ogg");
}

/*

*/
