/*
 * Copyright (C) 2001-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 *
 */

/*!
 * \file httpserv.c
 * \brief HTTP interface.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#ifndef MY_FSDEV
#define MY_FSDEV        devUrom
#endif

#ifdef MY_FSDEV_NAME
#define MY_HTTPROOT     MY_FSDEV_NAME ":/"
#endif

#include <cfg/os.h>
#include <cfg/memory.h>

#include "config.h"

#ifdef USE_HTTPSERVER

#ifndef HTTPD_THREAD_STACK
#define HTTPD_THREAD_STACK  2048
#endif

#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <time.h>

#include <dev/board.h>
#include <dev/urom.h>
#include <dev/vscodec.h>
#include <dev/watchdog.h>

#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <netdb.h>

#include <sys/confos.h>
#include <sys/confnet.h>
#include <sys/thread.h>
#include <sys/timer.h>
#include <sys/heap.h>
#include <sys/bankmem.h>
#include <sys/socket.h>
#include <sys/version.h>

#include <pro/ssi.h>
#include <pro/httpd.h>

#include "logmsg.h"
#include "utils.h"
#include "favlist.h"
#include "httpserv.h"

/*! \brief Dummy gain.
 *
 * Used to determine volume field that had been left empty.
 * We can't use zero.
 */
#define EMPTY_GAIN  999

extern void *__heap_start;
//extern void *_etext;

int h_timevalid;


/* Service thread counter. */
static int httpd_tc;

static int StartServiceThread(void);


/*!
 * \brief Create table of favorites.
 */
static int CgiStationFavorites(FILE * stream, REQUEST * req)
{
    int i;
    int row;
    int count = 0;
    char *value;
    char *id = NULL;
    int action = 0;
    count = NutHttpGetParameterCount(req);
    while (count--) {
        value = NutHttpGetParameterValue(req, count);
        if (strcmp(value, "edit") == 0) {
            action = 1;
            id = NutHttpGetParameterName(req, count);
        } else if (strcmp(value, "add new Favorite") == 0) {
            action = 2;
        }
    }
    if (action == 1 && id) {
        int idx = atoi(id);

        fputs("<tr class=\"tr1\"><td>Station Name</td>", stream);
        fprintf(stream,
                "<td align=\"left\"><INPUT type=\"hidden\" value=\"saveedit\" name=\"%s\"><INPUT type=\"text\" name=\"sname\" value=\"%s\" class=\"inputl\"></td>",
                id, favlist[idx].rs_name);
        fputs("</tr><tr class=\"tr2\"><td>Station URL's</td>", stream);
        fprintf(stream, "<td align=\"left\"><TEXTAREA class=\"inputl\" name=\"surl\" style=\"height:100;\">");
        for (i = 0; i < favlist[idx].rs_streams; i++) {
            fprintf(stream, "%s\n", favlist[idx].rs_uri[i]);
        }
        fprintf(stream, "</TEXTAREA></td>");
        fputs
            ("</tr><tr class=\"tr1\"><td>Save Settings</td><td align=\"right\"><INPUT type=\"submit\" value=\"Apply\" class=\"submit\"></td></tr>",
             stream);
    } else if (action == 2) {
        fputs("<tr class=\"tr1\"><td>Station Name</td>", stream);
        fputs
            ("<td align=\"left\"><INPUT type=\"hidden\" value=\"savenew\" name=\"0\"><INPUT type=\"text\" name=\"sname\" class=\"inputl\"></td>",
             stream);
        fputs("</tr><tr class=\"tr2\"><td>Station URL's</td>", stream);
        fputs("<td align=\"left\"><TEXTAREA class=\"inputl\" name=\"surl\" style=\"height:100;\"></TEXTAREA></td>", stream);
        fputs
            ("</tr><tr class=\"tr1\"><td>Save Settings</td><td align=\"right\"><INPUT type=\"submit\" name=\"func\" value=\"Apply\" class=\"submit\"></td></tr>",
             stream);
    } else {
        for (i = LAST_FAVORITE + 1, row = 0; i < MAX_FAVORITES; i++) {
            if (favlist[i].rs_name == NULL) {
                continue;
            }
            if (row & 1) {
                fputs("<tr class=\"tr1\">\r\n", stream);
            } else {
                fputs("<tr class=\"tr2\">\r\n", stream);
            }
            row++;

            fprintf(stream, "<td>%s</td>\r\n", favlist[i].rs_name);
            fprintf(stream, "<td><INPUT type=\"submit\" name=\"%d\" value=\"edit\" class=\"submit\"></td>\r\n", i);
            fprintf(stream,
                    "<td><INPUT type=\"submit\" name=\"%d\" value=\"delete\" class=\"submit\" onClick=\"return confirm('Are you sure you want to delete %s ?');\"></td>\r\n",
                    i, favlist[i].rs_name);
            fprintf(stream, "<td><INPUT type=\"submit\" name=\"%d\" value=\"play now\" class=\"submit\"></td>\r\n", i);
            fputs("</tr>\r\n", stream);
        }
        if (row & 1) {
            fputs("<tr class=\"tr1\">\r\n", stream);
        } else {
            fputs("<tr class=\"tr2\">\r\n", stream);
        }
        fputs
            ("<td colspan=\"4\" align=\"right\"><INPUT type=\"submit\" name=\"0\" value=\"add new Favorite\" class=\"submit\"></td>",
             stream);
        fputs("</tr>\r\n", stream);
    }
    fflush(stream);

    return 0;
}

static int CgiShoutCastGenres(FILE * stream, REQUEST * req)
{
    int i;
    int count;
    char *name;
    char *genre;
    char *getgenre = NULL;
    count = NutHttpGetParameterCount(req);
    for (i = 0; i < count; i++) {
        name = NutHttpGetParameterName(req, i);
        if (strcmp(name, "genre") == 0) {
            getgenre = NutHttpGetParameterValue(req, i);
            break;
        }
    }
#if 0
    if (getgenre == NULL) {
        fprintf(stream, "<OPTION SELECTED id=\"Top500\">Top500</OPTION>\r\n");
    } else {
        fprintf(stream, "<OPTION id=\"Top500\">Top500</OPTION>\r\n");
    }
#else
    if (getgenre == NULL) {
        fprintf(stream, "<OPTION SELECTED id=\"TopTen\">TopTen</OPTION>\r\n");
    } else {
        fprintf(stream, "<OPTION id=\"Top500\">TopTen</OPTION>\r\n");
    }
#endif
    for (i = 0;; i++) {
        genre = ShoutCastGetGenre(i);
        if (genre == NULL) {
            break;
        }
        if (strcmp(getgenre, genre) == 0) {
            fprintf(stream, "<OPTION SELECTED id=\"%s\">%s</OPTION>\r\n", genre, genre);
        } else {
            fprintf(stream, "<OPTION id=\"%s\">%s</OPTION>\r\n", genre, genre);
        }
    }
    fflush(stream);

    return 0;
}

static int CgiShoutCastStations(FILE * stream, REQUEST * req)
{
    int count;
    char *name;
    char *genre = NULL;
    int i;

    count = NutHttpGetParameterCount(req);
    for (i = 0; i < count; i++) {
        name = NutHttpGetParameterName(req, i);
        if (strcmp(name, "genre") == 0) {
            genre = NutHttpGetParameterValue(req, i);
            break;
        }
    }
    for (i = 0;; i++) {
        if (genre) {
            name = ShoutCastGetStationName(genre, i);
        } else {
            name = ShoutCastGetStationName("Top500", i);
        }
        if (name == NULL) {
            break;
        }
        if (i & 1) {
            fputs("<tr class=\"tr1\">\r\n", stream);
        } else {
            fputs("<tr class=\"tr2\">\r\n", stream);
        }
        fprintf(stream, "<td>%s</td>\r\n", name);
        fprintf(stream, "<td><INPUT type=\"submit\" name=\"%d\" value=\"play now\" class=\"submit\"></td>\r\n", i);
        fprintf(stream, "<td><INPUT type=\"submit\" name=\"%d\" value=\"add to favorites\" class=\"submit\"></td>\r\n", i);
        fputs("</tr>\r\n", stream);
    }
    fflush(stream);

    return 0;
}

static int CgiShoutCastControl(FILE * stream, REQUEST * req)
{
    int count;
    char *name;
    char *value;
    int action = 0;
    int idx = 0;

    count = NutHttpGetParameterCount(req);
    while (count--) {
        value = NutHttpGetParameterValue(req, count);
        if (strcmp(value, "add to favorites") == 0) {
            action = 1;
        }
        if (strcmp(value, "play now") == 0) {
            action = 2;
        }
        if (action) {
            if ((name = NutHttpGetParameterName(req, count)) == NULL) {
                action = 0;
            } else {
                idx = atoi(name);
                break;
            }
        }
    }
    if (action == 1) {
        ShoutCastAddStation(idx, MAX_FAVORITES);
        webradio.wr_favupd = 10;
    } else if (action == 2) {
        ShoutCastAddStation(idx, TOP_FAVORITE);
    }
    return 0;
}

static void AddStationUris(int idx, CONST char *str)
{
    char *uri = malloc(256);
    CONST char *cp;
    char *up;

    if (uri) {
        cp = str;
        while (*cp) {
            while (*cp && *cp < ' ') {
                cp++;
            }
            up = uri;
            while (*cp) {
                if (*cp < ' ') {
                    break;
                }
                *up++ = *cp++;
            }
            *up = 0;
            if (strlen(uri)) {
                LogMsg(LOG_HTTPD, "Add '%s'\n", uri);
                FavListSet(idx, NULL, uri);
            }
        }
        free(uri);
    }
}

static int CgiFavoritesControl(FILE * stream, REQUEST * req)
{
    int count;
    char *name;
    char *value;
    int action = 0;
    int idx = 0;
    char *sname = NULL;
    char *surl = NULL;

    count = NutHttpGetParameterCount(req);
    while (count--) {
        name = NutHttpGetParameterName(req, count);
        value = NutHttpGetParameterValue(req, count);
        if (strcmp(name, "sname") == 0) {
            sname = value;
        } else if (strcmp(name, "surl") == 0) {
            surl = value;
        } else if (strcmp(value, "saveedit") == 0) {
            action = 1;
            idx = atoi(name);
        } else if (strcmp(value, "delete") == 0) {
            action = 2;
            idx = atoi(name);
        } else if (strcmp(value, "play now") == 0) {
            action = 3;
            idx = atoi(name);
        } else if (strcmp(value, "savenew") == 0) {
            action = 4;
        }
    }
    if (action == 1) {
        /* Save existing. */
        idx = FavListSet(idx, sname, NULL);
        AddStationUris(idx, surl);
        webradio.wr_favupd = 2;
    } else if (action == 2) {
        FavListSet(idx, NULL, NULL);
        webradio.wr_favupd = 20;
    } else if (action == 3) {
        FavListCopy(idx, TOP_FAVORITE);
    } else if (action == 4) {
        /* Add new. */
        idx = FavListSet(MAX_FAVORITES, sname, NULL);
        AddStationUris(idx, surl);
        webradio.wr_favupd = 2;
    }
    return 0;
}

static int CgiSettings(FILE * stream, REQUEST * req)
{
    int count;
    char *name;
    char *value = NULL;
    char *hostname = NULL;
    char *prox = NULL;
    char *dhcp = NULL;
    char *ip = NULL;
    char *mask = NULL;
    char *gateway = NULL;
    char *dns0 = NULL;
    char *dns1 = NULL;
    char *func = NULL;
    u_long addr;
    int i;
    int upd_os = 0;
    int upd_net = 0;
    int upd_boot = 0;
    int upd_radio = 0;

    count = NutHttpGetParameterCount(req);
    for (i = 0; i < count; i++) {
        name = NutHttpGetParameterName(req, i);
        if (strcmp(name, "value") == 0) {
            value = NutHttpGetParameterValue(req, i);
            break;
        } else if (strcmp(name, "hostname") == 0) {
            hostname = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "proxy") == 0) {
            prox = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "dhcp") == 0) {
            dhcp = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "ip") == 0) {
            ip = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "mask") == 0) {
            mask = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "gateway") == 0) {
            gateway = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "dns0") == 0) {
            dns0 = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "dns1") == 0) {
            dns1 = NutHttpGetParameterValue(req, i);
        } else if (strcmp(name, "func") == 0) {
            func = NutHttpGetParameterValue(req, i);
        }
    }
    if (func) {
        if (strcmp(func, "Apply") == 0) {
            if (hostname) {
                strncpy(confos.hostname, hostname, sizeof(confos.hostname) - 1);
                upd_os = 1;
            }
            if (prox) {
                char *cp = prox;

                while (*cp && *cp <= ' ') {
                    cp++;
                }
                if (strlen(cp)) {
                    strncpy(proxy.proxy_host, cp, sizeof(proxy.proxy_host) - 1);
                    if ((cp = strrchr(proxy.proxy_host, ':')) != NULL) {
                        *cp++ = 0;
                        proxy.proxy_port = atoi(cp);
                    }
                }
                else {
                    memset(&proxy, 0, sizeof(proxy));
                }
                upd_radio = 1;
            }
            if (dhcp && strcmp(dhcp, "on") == 0) {
                confnet.cdn_cip_addr = 0;
                upd_net = 1;
                upd_boot = 1;
            }
            else {
                if (ip) {
                    addr = inet_addr(ip);
                    if ((long)addr != -1) {
                        confnet.cdn_cip_addr = addr;
                        upd_net = 1;
                        upd_boot = 1;
                    }
                }
                if (mask) {
                    addr = inet_addr(mask);
                    if ((long)addr != -1) {
                        confnet.cdn_ip_mask = addr;
                        upd_net = 1;
                        upd_boot = 1;
                    }
                }
                if (gateway) {
                    addr = inet_addr(gateway);
                    if ((long)addr != -1) {
                        confnet.cdn_gateway = addr;
                        upd_net = 1;
                        upd_boot = 1;
                    }
                }
                if (dns0) {
                    addr = inet_addr(dns0);
                    if ((long)addr != -1) {
                        webradio.wr_pridns = addr;
                        upd_radio = 1;
                        upd_boot = 1;
                    }
                }
                if (dns1) {
                    addr = inet_addr(dns1);
                    if ((long)addr != -1) {
                        webradio.wr_secdns = addr;
                        upd_radio = 1;
                        upd_boot = 1;
                    }
                }
            }
            if (upd_os) {
                LogMsg(LOG_HTTPD, "Saving OS\n");
                NutSaveConfig();
            }
            if (upd_net) {
                LogMsg(LOG_HTTPD, "Saving Net\n");
                NutNetSaveConfig();
            }
            if (upd_radio) {
                ConfigSave();
            }
            if (upd_boot) {
                LogMsg(LOG_HTTPD, "Reboot\n");
                webradio.wr_reboot = 3;
            }
        } 
        else if (strcmp(func, "Erase") == 0) {
            u_char mac[6];

            memset(&confos, 0, sizeof(confos));
            NutSaveConfig();
            memcpy(mac, confnet.cdn_mac, sizeof(mac));
            memset(&confnet, 0, sizeof(confnet));
            memcpy(confnet.cdn_mac, mac, sizeof(confnet.cdn_mac));
            NutNetSaveConfig();
            ConfigResetFactory();
            FavListResetFactory();
            ConfigSave();
            FavListSave();
            webradio.wr_reboot = 2;
        }
    }
    else if (value) {
        if (strcmp(value, "hostname") == 0) {
            fprintf(stream, "%s", confos.hostname);
        } else if (strcmp(value, "dhcp") == 0) {
            if (confnet.cdn_cip_addr == 0) {
                fputs(" checked=\"checked\"", stream);
            }
        } else if (strcmp(value, "proxy") == 0) {
            if (proxy.proxy_port) {
                fputs(proxy.proxy_host, stream);
                if (proxy.proxy_port != 80) {
                    fprintf(stream, ":%u", proxy.proxy_port);
                }
            }
        } else if (strcmp(value, "ip") == 0) {
            if (confnet.cdn_cip_addr) {
                fputs(inet_ntoa(confnet.cdn_cip_addr), stream);
            } else {
                fputs(inet_ntoa(confnet.cdn_ip_addr), stream);
            }
        } else if (strcmp(value, "mask") == 0) {
            fputs(inet_ntoa(confnet.cdn_ip_mask), stream);
        } else if (strcmp(value, "gateway") == 0) {
            fputs(inet_ntoa(confnet.cdn_gateway), stream);
        } else if (strcmp(value, "dns0") == 0) {
            NutDnsGetConfig2(NULL, NULL, &addr, NULL);
            fputs(inet_ntoa(addr), stream);
        } else if (strcmp(value, "dns1") == 0) {
            NutDnsGetConfig2(NULL, NULL, NULL, &addr);
            fputs(inet_ntoa(addr), stream);
        }
    }
    return 0;
}

static int CgiVars(FILE * stream, REQUEST * req)
{
    int count;
    char *name;
    char *cgivar = NULL;
    int i;

    count = NutHttpGetParameterCount(req);
    for (i = 0; i < count; i++) {
        name = NutHttpGetParameterName(req, i);
        if (strcmp(name, "var") == 0) {
            cgivar = NutHttpGetParameterValue(req, i);
            break;
        }
    }

    if (cgivar == NULL) {
        fputs("Error", stream);
    } else if (strcmp(cgivar, "MetaTitle") == 0 || strcmp(cgivar, "StreamTitle") == 0) {
        if (webradio.wr_rip) {
            SHOUTCASTINFO *sci = (SHOUTCASTINFO *) webradio.wr_rip->ri_bcast;
            if (sci && sci->sci_metatitle) {
                fputs(sci->sci_metatitle, stream);
            }
        }
    } else if (strcmp(cgivar, "CPUClock") == 0) {
        fprintf(stream, "%lu kHz", NutGetCpuClock() / 1000);
    } else if (strcmp(cgivar, "MasterClock") == 0) {
#if defined(AT91_PLL_MAINCK)
        fprintf(stream, "%lu kHz", At91GetMasterClock() / 1000);
#endif
    } else if (strcmp(cgivar, "TotalMemoryUsed") == 0) {
        fprintf(stream, "%u kBytes", (u_int) (NUTMEM_SIZE - NutHeapAvailable()) / 1024);
    } else if (strcmp(cgivar, "StationName") == 0) {
        if (webradio.wr_sip) {
            fputs(webradio.wr_sip->si_name, stream);
        }
    } else if (strcmp(cgivar, "StationGenre") == 0) {
        if (webradio.wr_sip && webradio.wr_sip->si_genre) {
            fputs(webradio.wr_sip->si_genre, stream);
        }
    } else if (strcmp(cgivar, "StationBitrate") == 0) {
        if (webradio.wr_sip) {
            fprintf(stream, "%u kBit", webradio.wr_sip->si_bitrate);
        }
    } else if (strcmp(cgivar, "BufferContents") == 0) {
        fprintf(stream, "%lu Bytes", NutSegBufUsed());
    } else if (strcmp(cgivar, "StreamURL") == 0) {
        if (webradio.wr_rip) {
            SHOUTCASTINFO *sci = (SHOUTCASTINFO *) webradio.wr_rip->ri_bcast;
            if (sci->sci_metaurl) {
                fprintf(stream, "%s", sci->sci_metaurl);
            }
        }
    } else if (strcmp(cgivar, "MetadataInterval") == 0) {
        if (webradio.wr_rip) {
            SHOUTCASTINFO *sci = (SHOUTCASTINFO *) webradio.wr_rip->ri_bcast;
            fprintf(stream, "%lu Bytes", sci->sci_metaint);
        }
    } else if (strcmp(cgivar, "NutVersion") == 0) {
        fputs(NutVersionString(), stream);
    } else if (strcmp(cgivar, "AppVersion") == 0) {
        fputs(VERSION, stream);
    } else {
        fputs("Unknown", stream);
    }
    fflush(stream);
    return 0;
}

/*! \fn Service(void *arg)
 * \brief HTTP service thread.
 *
 * The endless loop in this thread waits for a client connect,
 * processes the HTTP request and disconnects. Nut/Net doesn't
 * support a server backlog. If one client has established a
 * connection, further connect attempts will be rejected.
 * Typically browsers open more than one connection in order
 * to load images concurrently. So we run this routine by
 * several threads.
 *
 */
THREAD(Service, arg)
{
    TCPSOCKET *sock;
    FILE *stream;

    LogMsg(LOG_HTTPD, "Started HTTP daemon\n");
    for (;;) {
        /* Create a socket. */
        if ((sock = NutTcpCreateSocket()) == 0) {
            LogMsg(LOG_WARN, "No sockets\n");
            NutSleep(1000);
            continue;
        }

        /* Set socket options. Silently ignore any error. */
#ifdef HTTPD_MAX_SEGSIZE
        {
            u_short mss = HTTPD_MAX_SEGSIZE;
            NutTcpSetSockOpt(sock, TCP_MAXSEG, &mss, sizeof(mss));
        }
#endif
#ifdef HTTPD_TCP_BUFSIZE
        {
            u_short tcpbufsiz = HTTPD_TCP_BUFSIZE;
            NutTcpSetSockOpt(sock, SO_RCVBUF, &tcpbufsiz, sizeof(tcpbufsiz));
        }
#endif
#ifdef HTTPD_TCP_TIMEOUT
        {
            u_long tmo = HTTPD_TCP_TIMEOUT;
            NutTcpSetSockOpt(sock, SO_RCVTIMEO, &tmo, sizeof(tmo));
        }
#endif

        LogMsg(LOG_HTTPD, "Waiting for HTTP client\n");
        stream = TcpStreamAccept(sock, HTTPD_TCP_PORT, "r+b");
        if (stream) {
            LogMsg(LOG_HTTPD, "Connected HTTP client\n");
#ifdef USE_DYNAMIC_THREADS
            /* Start a new thread and let it run. */
            StartServiceThread();
            NutSleep(1);
#endif
            /* Let the Nut/OS library process the client's requests. */
            NutHttpProcessRequest(stream);
            LogMsg(LOG_HTTPD, "Disconnecting HTTP client\n");
            fclose(stream);
        }
        NutTcpCloseSocket(sock);

#ifdef USE_DYNAMIC_THREADS
        /* If enough threads are running, stop this one. */
        if (httpd_tc >= HTTPD_MIN_THREADS) {
            httpd_tc--;
            LogMsg(LOG_HTTPD, "Stop HTTP daemon\n");
            NutThreadExit();
        }
#endif
    }
}

/*
 * Start a HTTP daemon thread.
 */
static int StartServiceThread(void)
{
#ifdef USE_DYNAMIC_THREADS
    if (httpd_tc >= HTTPD_MAX_THREADS) {
        return 0;
    }
#endif

    if (NutThreadCreate("httpd", Service, NULL, HTTPD_SERVICE_STACK) == NULL) {
        return -1;
    }
    httpd_tc++;

    return 0;
}

/*!
 * \brief HTTP Server Start.
 */
int HttpServerStart(void)
{
    int i;

    /*
     * Register our device for the file system.
     */
    if (NutRegisterDevice(&MY_FSDEV, 0, 0)) {
        LogMsg(LOG_ERROR, "No HTTP mount\n");
        return -1;
    }

#ifdef MY_HTTPROOT
    /* Register root directory. */
    LogMsg(LOG_HTTPD, "Register root dir '" MY_HTTPROOT "'\n");
    if (NutRegisterHttpRoot(MY_HTTPROOT)) {
        LogMsg(LOG_ERROR, "No HTTP root\n");
        return -1;
    }
#endif
    /*
     * Register SSI handler
     */
    NutRegisterSsi();

    /*
     * Register our CGIs.
     */
    NutRegisterCgi("f_control.cgi", CgiFavoritesControl);
    NutRegisterCgi("favorites.cgi", CgiStationFavorites);
    NutRegisterCgi("genres.cgi", CgiShoutCastGenres);
    NutRegisterCgi("stations.cgi", CgiShoutCastStations);
    NutRegisterCgi("sc_control.cgi", CgiShoutCastControl);
    NutRegisterCgi("vars.cgi", CgiVars);
    NutRegisterCgi("settings.cgi", CgiSettings);

    /*
     * Protect the admin directory with user and password.
     */
    NutRegisterAuth("admin", "admin:admin");

    /*
     * Start four server threads.
     */
    LogMsg(LOG_HTTPD, "Starting %d HTTP daemons\n", HTTPD_MIN_THREADS);
    for (i = 0; i < HTTPD_MIN_THREADS; i++) {
        if (StartServiceThread()) {
            LogMsg(LOG_ERROR, "Thread start failed\n");
            return -1;
        }
    }

    /* Let started threads take over. */
    NutSleep(1);

    return 0;
}

#endif /* USE_HTTPSERVER */
