/*
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file receiver.c
 * \brief Receiver.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <cfg/os.h>
#include <cfg/clock.h>
#include <dev/board.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>

#include <sys/version.h>
#include <sys/confnet.h>
#include <sys/atom.h>
#include <sys/heap.h>
#include <sys/thread.h>
#include <sys/timer.h>
#include <sys/event.h>
#include <sys/socket.h>

#if defined(USE_SOFTWARE_CODEC)
#include <hxmp3/mp3dec.h>
#endif

#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <pro/dhcp.h>

#include "config.h"
#include "logmsg.h"
#include "receiver.h"
#include "shoutcast.h"

/*!
 * \brief Create stream receiver instance with the specified plug-in.
 *
 * \param plugin Pointer to a receiver plug-in.
 *
 * \return Pointer to the receiver info structure or NULL, if the creation
 *         failed.
 */
RECEIVERINFO * ReceiverCreate(RECEIVERPLUGIN *plugin)
{
    RECEIVERINFO *rip;

    /* Allocate and initialize the receiver info structure. */
    if ((rip = malloc(sizeof(RECEIVERINFO))) != NULL) {
        memset(rip, 0, sizeof(RECEIVERINFO));
        rip->ri_rpp = plugin;
        /* Associate a new player plug-in instance. */
        if ((*plugin->rp_create) (rip)) {
            free(rip);
            rip = NULL;
        }
    }
    return rip;
}

/*!
 * \brief Change receiver plug-in status.
 *
 * \param rip Points to the receiver information structure.
 * \param sst Command status to set, either RSTAT_STOP or \ref RSTAT_START.
 * \param xst Expected final status, \ref RSTAT_IDLE, \ref RSTAT_BUFFERING 
 *            or \ref RSTAT_RUNNING. The last two may be or'ed together.
 *
 * \return 0 on success, -1 otherwise.
 */
static int ReceiverPlugInControl(RECEIVERINFO *rip, u_int sst, u_int xst)
{
    int retries = 10;

    while ((rip->ri_status & xst) == 0) {
        rip->ri_status |= sst;
        NutEventPost(&rip->ri_cmdevt);
        NutEventWait(&rip->ri_stsevt, 500);
        if (retries-- <= 0) {
            LogMsg(LOG_ERROR, "No receiver control (%02x->%02x->%02x)\n", sst, xst, rip->ri_status);
            return -1;
        }
    }
    return 0;
}

/*!
 * \brief Retrieve the current receiver status.
 *
 * \param rip Pointer to the receiver information structure.
 * 
 * \return Receiver status.
 */
u_int ReceiverStatus(RECEIVERINFO *rip)
{
    return rip->ri_status;
}

/*!
 * \brief Stop receiver.
 *
 * \param rip Receiver to stop.
 *
 * \return 0 on success, -1 otherwise.
 */
int ReceiverStop(RECEIVERINFO *rip)
{
    return ReceiverPlugInControl(rip, RSTAT_STOP, RSTAT_IDLE);
}

/*!
 * \brief Start receiver for the connected stream.
 *
 * \param ritab Table of available receivers.
 * \param sip   Information structure of the connected station .
 *
 * \return Pointer to the receiver entry that had been started.
 *         NULL is returned in case any error.
 */
RECEIVERINFO *ReceiverStart(RECEIVERINFO *ritab[], STATIONINFO *sip)
{
    RECEIVERINFO *rip;
    int i;

    for (i = 0; ritab[i]; i++) {
        rip = ritab[i];
        if ((*rip->ri_rpp->rp_setup)(rip, sip) == 0) {
            /* Found the right receiver. Make sure it has stopped. */
            if (ReceiverStop(rip) == 0) {
                /* Now start this receiver. */
                rip->ri_sip = sip;
                if (ReceiverPlugInControl(rip, RSTAT_START, RSTAT_BUFFERING | RSTAT_RUNNING) == 0) {
                    /* Success. */
                    return rip;
                }
            }
        }
    }
    LogMsg(LOG_WARN, "Receiver start failed\n");

    return NULL;
}

