#ifndef _RECEIVER_H_
#define _RECEIVER_H_

/*
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file receiver.h
 * \brief Audio data receiver.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <sys/socket.h>
#include "config.h"
#include "station.h"

/*! \name Receiver Status */
/*@{*/
/*! \brief Receiver is idle. */
#define RSTAT_IDLE      0x0001
/*! \brief Request to start receiver. */
#define RSTAT_START     0x0002
/*! \brief Receiver is running. */
#define RSTAT_BUFFERING 0x0004
/*! \brief Receiver is running. */
#define RSTAT_RUNNING   0x0008
/*! \brief Request to stop receiver. */
#define RSTAT_STOP      0x0010
/*@}*/

typedef struct _RECEIVERINFO RECEIVERINFO;

/*!
 * \brief Receiver plug-in reference structure.
 *
 * A variable with this structure is provided by the plug-in.
 */
typedef struct _RECEIVERPLUGIN {
    int (*rp_create) (RECEIVERINFO * rip);
    int (*rp_setup) (RECEIVERINFO * rip, STATIONINFO *sip);
} RECEIVERPLUGIN;

/*!
 * \brief Receiver information structure.
 *
 * An instance of this structure is created when calling ReceiverCreate().
 */
struct _RECEIVERINFO {
    /*! \brief Decoder device handle. */
    int ri_decoder;
    /*! \brief Current status of the receiver.
     *
     * - \ref RSTAT_IDLE if stopped.
     * - \ref RSTAT_START if currently starting.
     * - \ref RSTAT_BUFFERING if filling receive buffer.
     * - \ref RSTAT_RUNNING if started.
     * - \ref RSTAT_STOP if currently stopping.
     */     
    u_int ri_status;
    /*! \brief Receiver plug-in command event queue.
     *
     * An event is posted to this queue whenever a status change is requested.
     */
    HANDLE ri_cmdevt;
    /*! \brief Receiver plug-in status event queue.
     *
     * An event is posted by the receiver plug-in to this queue whenever a 
     * status change took place.
     */
    HANDLE ri_stsevt;
    /*! \brief Pointer to the station information structure.
     */
    STATIONINFO *ri_sip;
    /*! \brief Pointer to the plug-in reference structure.
     */
    RECEIVERPLUGIN *ri_rpp;
    /*! \brief Receiver plug-in local data.
     *
     * Can be used by the receiver plug-in to store local data.
     */
    void *ri_bcast;
};

__BEGIN_DECLS
/* Prototypes */

extern RECEIVERINFO * ReceiverCreate(RECEIVERPLUGIN *plugin);
extern RECEIVERINFO * ReceiverStart(RECEIVERINFO *ritab[], STATIONINFO *sip);
extern int ReceiverStop(RECEIVERINFO *rip);
extern u_int ReceiverStatus(RECEIVERINFO *rip);

__END_DECLS
/* End of prototypes */
#endif
