/*
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file shoutcast.c
 * \brief SHOUTcast communication routines.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <cfg/os.h>
#include <cfg/clock.h>
#include <dev/board.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>

#include <sys/version.h>
#include <sys/confnet.h>
#include <sys/atom.h>
#include <sys/heap.h>
#include <sys/thread.h>
#include <sys/timer.h>
#include <sys/event.h>
#include <sys/socket.h>

#if defined(USE_SOFTWARE_CODEC)
#include <hxmp3/mp3dec.h>
#endif

#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <pro/dhcp.h>
#include <netdb.h>

#include <pro/uxml.h>
#include <dev/vscodec.h>

#include "config.h"
#include "logmsg.h"
#include "favlist.h"
#include "utils.h"
#include "userif.h"
#include "xmlserv.h"
#include "shoutcast.h"

#ifndef SHOUTCAST_THREAD_STACK
#ifdef AT91SAM7X_EK
#define SHOUTCAST_THREAD_STACK  1024
#else
#define SHOUTCAST_THREAD_STACK  2048
#endif
#endif

#define SHOUTCAST_QLIST "www.shoutcast.com/sbin/newxml.phtml"
#define SHOUTCAST_QTUNE "www.shoutcast.com/sbin/tunein-station.pls"

static char station_genre[32];
static UXML_NODE *genre_tree;

#define ENCTYP_MPEG     1
#define ENCTYP_AAC      2
#define ENCTYP_AACP     3
#define ENCTYP_OGG      4
#define ENCTYP_WMA      5


typedef struct {
    char *s_name;   /* Station name. */
    long s_id;      /* SHOUTcast ID. */
    long s_br;      /* Bitrate. */
    int  s_enc;     /* Encoding. */
} STATION_INFO;

static int station_cnt;
static STATION_INFO *station_list;

static FILE *OpenTcpStream(TCPSOCKET *sock, CONST char *host, u_short port)
{
    u_long rx_to = MAX_TCPRCV_WAIT;

    /* Set socket options. Failures are ignored. */
    NutTcpSetSockOpt(sock, SO_RCVTIMEO, &rx_to, sizeof(rx_to));

    /* Connect the stream server. */
    LogMsg(LOG_STATION, "Connecting %s:%u\n", host, port);
    if (TcpHostConnect(sock, host, port)) {
        return NULL;
    }
    LogMsg(LOG_SHOUTCAST, "Connected\n");
    /* Associate a binary stream with the socket. */
    return _fdopen((int) ((uptr_t) sock), "r+b");
}

static int SendHttpRequest(FILE *stream, HTTP_SCHEME *schm)
{
    fputs("GET ", stream);
    if (proxy.proxy_port) {
        fprintf(stream, "http://%s", schm->schm_uri);
    }
    fprintf(stream, "/%s HTTP/1.1\r\n", schm->schm_path);
    fprintf(stream, "Host: %s\r\n", schm->schm_host);
    fputs("User-Agent: WinampMPEG/2.7\r\n" /* */
        "Icy-MetaData: 1\r\n"  /* */
        "Connection: close\r\n\r\n", stream);
    fflush(stream);

    return 0;
}

/*!
 * \brief Retrieve a genre list from the SHOUTcast server.
 */
static int ShoutCastGetGenreList(void)
{
    int rc = -1;
    TCPSOCKET *sock;
    HTTP_SCHEME *schm;
    FILE *stream;
    char *f_tags[] = { "genre", NULL };
    char *f_attribs[] = { "name", NULL };

    if ((schm = HttpSchemeParse(SHOUTCAST_QLIST)) == NULL) {
        return -1;
    }

    if ((sock = NutTcpCreateSocket()) != NULL) {
        if ((stream = OpenTcpStream(sock, schm->schm_host, schm->schm_portnum)) != NULL) {
            SendHttpRequest(stream, schm);
            if ((genre_tree = UxmlParseStream(stream, f_tags, f_attribs)) != NULL) {
                rc = 0;
            }
            fclose(stream);
        }
        NutTcpCloseSocket(sock);
    }
    HttpSchemeRelease(schm);

    return rc;
}

/*
 * List sort helper.
 */
int station_cmp(CONST void *ip1, CONST void *ip2)
{
    NutThreadYield();
    return strcasecmp(((STATION_INFO *)ip1)->s_name, ((STATION_INFO *)ip2)->s_name);
}

/*!
 * \brief Retrieve station list of a specified genre from SHOUTcast.
 */
static int ShoutCastGetStationList(CONST char *genre)
{
    int rc = -1;
    UXML_NODE *station_tree = NULL;
    TCPSOCKET *sock;
    HTTP_SCHEME *schm;
    FILE *stream;
    char *uri;
    char *f_tags[] = { "station", NULL };
    char *f_attribs[] = { "name", "mt", "id", "br", NULL };

    /* Release the current list. */
    if (station_list) {
        while (station_cnt) {
            station_cnt--;
            free(station_list[station_cnt].s_name);
        }
        free(station_list);
        station_list = NULL;
    }

    strcpy(station_genre, genre);

    /* Create the request string. */
    uri = malloc(sizeof(SHOUTCAST_QLIST) + strlen(genre) + sizeof("?genre="));
    strcpy(uri, SHOUTCAST_QLIST);
    strcat(uri, "?genre=");
    strcat(uri, genre);
    schm = HttpSchemeParse(uri);
    free(uri);
    if (schm == NULL) {
        return -1;
    }

    if ((sock = NutTcpCreateSocket()) != NULL) {
        if ((stream = OpenTcpStream(sock, schm->schm_host, schm->schm_portnum)) != NULL) {
            SendHttpRequest(stream, schm);
            if ((station_tree = UxmlParseStream(stream, f_tags, f_attribs)) != NULL) {
                rc = 0;
            }
            fclose(stream);
        }
        NutTcpCloseSocket(sock);
    }
    HttpSchemeRelease(schm);

    if (rc == 0) {
        UXML_ATTRIB *attr;
        UXML_NODE *node = station_tree;

        for (node = station_tree; node; node = node->xmln_next) {
            //Optional filter attr = node->xmln_attribs;
            station_cnt++;
        }
        LogMsg(LOG_SHOUTCAST, "\n%d Stations", station_cnt);
        if (station_cnt) {
            station_list = malloc(station_cnt * sizeof(STATION_INFO));
            memset(station_list, 0, station_cnt * sizeof(STATION_INFO));
            station_cnt = 0;
            for (node = station_tree; node; node = node->xmln_next) {
                attr = node->xmln_attribs;
                while (attr) {
                    // Repeat optional filter
                    if (strcasecmp(attr->xmla_name, "name") == 0) {
                        station_list[station_cnt].s_name = attr->xmla_value;
                        attr->xmla_value = NULL;
                    }
                    if (strcasecmp(attr->xmla_name, "id") == 0) {
                        station_list[station_cnt].s_id = atol(attr->xmla_value);
                    }
                    attr = attr->xmla_next;
                }
                if (station_list[station_cnt].s_name) {
                    station_cnt++;
                }
            }
            qsort(station_list, station_cnt, sizeof(STATION_INFO), station_cmp);
        }
        UxmlTreeDestroy(station_tree);
    }
    return rc;
}

char * ShoutCastGetGenre(int idx)
{
    UXML_NODE *node;
    UXML_ATTRIB *attr;

    if (genre_tree == NULL) {
        ShoutCastGetGenreList();
    }
    node = genre_tree;
    while (node) {
        attr = node->xmln_attribs;
        while (attr) {
            if (idx == 0) {
                return attr->xmla_value;
            }
            idx--;
            attr = attr->xmla_next;
        }
        node = node->xmln_next;
    }
    return NULL;
}

char * ShoutCastGetStationName(char *genre, int idx)
{
    if (strcmp(station_genre, genre)) {
        ShoutCastGetStationList(genre);
    }
    if (idx < station_cnt) {
        return station_list[idx].s_name;
    }
    return NULL;
}

/*!
 * \brief Query streams of a specified station and create a favorites list entry.
 *
 * \param idx SHOUTcast station list index.
 * \param pos Favorites list index.
 *
 * \return 0 on success, -1 otherwise.
 */
int ShoutCastAddStation(int idx, int pos)
{
    int rc = -1;
    TCPSOCKET *sock;
    HTTP_SCHEME *schm;
    FILE *stream;
    char *line;
    char *uri;
    int inlist;
    char *cp;

    if (idx >= station_cnt) {
        return -1;
    }
    uri = malloc(sizeof(SHOUTCAST_QTUNE) + 16);
    sprintf(uri, SHOUTCAST_QTUNE "?id=%ld", station_list[idx].s_id);
    schm = HttpSchemeParse(uri);
    free(uri);
    if (schm == NULL) {
        return -1;
    }
    if ((sock = NutTcpCreateSocket()) != NULL) {
        if ((stream = OpenTcpStream(sock, schm->schm_host, schm->schm_portnum)) != NULL) {
            SendHttpRequest(stream, schm);
            line = malloc(128);
            line[127] = 0;
            inlist = 0;
            pos = FavListSet(pos, station_list[idx].s_name, NULL);
            while (fgets(line, 127, stream)) {
                if (inlist) {
                    if ((cp = strchr(line, '\r')) == NULL) {
                        cp = strchr(line, '\n');
                    }
                    if (cp) {
                        *cp = 0;
                    }
                    cp = strchr(line, '=');
                    if (cp && strncmp(line, "File", 4) == 0) {
                        FavListSet(pos, NULL, cp + 8);
                        LogMsg(LOG_SHOUTCAST, "Added %s\n", cp + 8);
                    }
                }
                else if (strncmp(line, "[playlist]", sizeof("[playlist]") - 1) == 0) {
                    inlist = 1;
                }
            }
            free(line);
            fclose(stream);
            rc = 0;
        }
        NutTcpCloseSocket(sock);
    }
    HttpSchemeRelease(schm);

    return rc;
}

int ShoutCastGetPlayList(int id)
{
    TCPSOCKET *sock;
    u_long rx_to = MAX_TCPRCV_WAIT;
    int cr;
    int err = 0;
    char *line;
    int len;
    HTTP_SCHEME *schm;
    char *uri;
    int entries = 0;

    uri = malloc(sizeof(SHOUTCAST_QTUNE) + 16);
    sprintf(uri, SHOUTCAST_QTUNE "?id=%d", id);
    schm = HttpSchemeParse(uri);
    free(uri);
    if (schm == NULL) {
        return -1;
    }

    /* Create a TCP socket. */
    if ((sock = NutTcpCreateSocket()) == NULL) {
        HttpSchemeRelease(schm);
        return -1;
    }

    /* Set socket options. Failures are ignored. */
    NutTcpSetSockOpt(sock, SO_RCVTIMEO, &rx_to, sizeof(rx_to));

    /* Connect the stream server. */
    LogMsg(LOG_STATION, "Connecting %s:%u", schm->schm_host, schm->schm_portnum);
    cr = TcpHostConnect(sock, schm->schm_host, schm->schm_portnum);
    if (cr == 0) {
        LogMsg(LOG_SHOUTCAST, "[CNCTD]");

        /*
        * Send the HTTP request.
        */
        line = malloc(256);

        if (proxy.proxy_port) {
            /* A proxy requires an absolute URI. */
            strcpy(line, "GET http://");
            strcat(line, schm->schm_uri);
        }
        else {
            strcpy(line, "GET /");
            strcat(line, schm->schm_path);
        }
        strcat(line, " HTTP/1.1\r\n");
        err = TcpPutString(sock, line);

        if (err == 0) {
            /* HTTP 1.1 requires this. So just in case... */
            sprintf(line, "Host: %s\r\n", schm->schm_host);
            err = TcpPutString(sock, line);
            LogMsg(LOG_SHOUTCAST, "%s", line);
        }

        if (err == 0) {
            /* Some SHOUTcast servers seem to require a user-agent line. 
                "Eat chalk" to get in. */
            TcpPutString(sock, "User-Agent: WinampMPEG/2.7\r\n" /* */
                "Icy-MetaData: 1\r\n"  /* */
                "Connection: close\r\n\r\n");
        }
        while (err == 0) {
            if ((len = TcpGetLine(sock, line, 255)) < 0) {
                break;
            }
            if (entries) {
                if (strncmp(line, "File", 4) == 0) {
                    LogMsg(LOG_SHOUTCAST, "OK\n");
                }
            }
            else if (strncmp(line, "numberofentries=", 16) == 0) {
                entries = atoi(line + 16);
            }
        }
        LogMsg(LOG_SHOUTCAST, "Collected %d entries\n", entries);
    }
    NutTcpCloseSocket(sock);
    HttpSchemeRelease(schm);

    return err;
}

/*!
 * \brief Process embedded meta data.
 */
static int ProcessMetaData(TCPSOCKET * sock, SHOUTCASTINFO * sci, u_int *status)
{
    u_char blks = 0;
    u_int mlen;
    char *mbuf;
    char *mn1;
    char *mn2;
    char *md1;
    char *md2;

    /*
     * Wait for the lenght byte.
     */
    if (TcpGetBuffer(sock, (char *)&blks, 1, status)) {
        /* Status change or receive error. */
        return -1;
    }

    if (blks == 0) {
        /* Empty metadata block. */
        return 0;
    }
    if (blks > 32) {
        /* We are probably out of sync. */
        return -1;
    }
    mlen = (u_int)blks * 16;

    /*
     * Wait for the metadata block.
     */
    if ((mbuf = malloc(mlen + 1)) == 0) {
        return -1;
    }
    mbuf[mlen] = 0;
    if (TcpGetBuffer(sock, mbuf, mlen, status)) {
        /* Status change or receive error. */
        return -1;
    }

    //ClearMetaData();
    LogMsg(LOG_SHOUTCAST, "Meta=\"%s\"\n", mbuf);
    mn1 = mbuf;
    while (mn1) {
        if ((mn2 = strchr(mn1, ';')) != 0)
            *mn2++ = 0;
        if ((md1 = strchr(mn1, '=')) != 0) {
            *md1++ = 0;
            while (*md1 == ' ' || *md1 == '\'')
                md1++;
            if ((md2 = strrchr(md1, '\'')) != 0)
                *md2 = 0;
            if (strcasecmp(mn1, "StreamTitle") == 0) {
                if (sci->sci_metatitle) {
                    free(sci->sci_metatitle);
                }
                sci->sci_metatitle = strdup(md1);
                UserIfShowStatus(DIST_FORCE);
            } else if (strcasecmp(mn1, "StreamUrl") == 0) {
                if (sci->sci_metaurl) {
                    free(sci->sci_metaurl);
                }
                sci->sci_metaurl = strdup(md1);
            }
        }
        mn1 = mn2;
    }
    free(mbuf);

    return 0;
}

/*!
 * \brief SHOUTcast receiver thread.
 *
 * Reads audio data from a TCP socket and passes it to a ring buffer.
 * Meta data is parsed and removed from the audio stream.
 *
 * \param arg Pointer to the player info structure.
 */
THREAD(ShoutCastThread, arg)
{
    RECEIVERINFO *rip = (RECEIVERINFO *) arg;
    SHOUTCASTINFO *sci = (SHOUTCASTINFO *) rip->ri_bcast;
    STATIONINFO *sip;
    int rbytes;
    int sent;
    int got;
    char *tcpbuf;
    char *bp;
    time_t buffering;
    int timeouts;

    for (;;) {
        /* 
         * Idle loop. 
         */
        rip->ri_decoder = -1;
        for (;;) {
            memset(sci, 0, sizeof(SHOUTCASTINFO));
            rip->ri_status &= ~RSTAT_STOP;
            rip->ri_status |= RSTAT_IDLE;
            NutEventBroadcast(&rip->ri_stsevt);
            LogMsg(LOG_SHOUTCAST, "Receiver idle\n");

            /* Wait for start event. */
            NutEventWait(&rip->ri_cmdevt, 0);
            if (rip->ri_status & RSTAT_START) {
                if ((rip->ri_decoder = _open("audio0", _O_WRONLY | _O_BINARY)) != -1) {
                    rip->ri_status &= ~(RSTAT_START | RSTAT_IDLE);
                    break;
                }
                LogMsg(LOG_ERROR, "No codec device\n");
            }
        }

        sip = rip->ri_sip;
        sci->sci_metapos = sci->sci_metaint;
        /* Broadcast running event. */
        rip->ri_status |= RSTAT_BUFFERING;
        NutEventBroadcast(&rip->ri_stsevt);

        /* Set initial audio buffer watermarks and streaming start time. */
        {
            u_long sz;

            /* Get codec buffer size. */
            _ioctl(rip->ri_decoder, AUDIO_GET_PBSIZE, &sz);
            /* Use 1/3 as low watermark. */
            sz /= 3;
            _ioctl(rip->ri_decoder, AUDIO_SET_PBWLOW, &sz);
            /* Use 2/3 as high watermark. */
            sz *= 2;
            _ioctl(rip->ri_decoder, AUDIO_SET_PBWHIGH, &sz);
        }

        /* So much copying! This is the disadvantage of the independent audio driver. */
        tcpbuf = malloc(4096);

        /* Start with buffering mode. */
        time(&buffering);

        /* Clear timeout counter. */
        timeouts = 0;

        /*
         * This loop reads audio data from the SHOUTcast server and passes
         * it to the audio decoder.
         */
        while ((rip->ri_status & RSTAT_STOP) == 0) {
            rbytes = 4096;
            /* Do not read beyond meta data. */
            if (sci->sci_metaint && rbytes > sci->sci_metapos) {
                rbytes = sci->sci_metapos;
            }

            /* Receive audio data from the Internet radio server. */
            if ((got = NutTcpReceive(sip->si_sock, tcpbuf, rbytes)) < 0) {
                LogMsg(LOG_SHOUTCAST, "Receiver error %d\n", NutTcpError(sip->si_sock));
                break;
            }

            /* Stop receiver on too many timeouts. */
            if (got == 0) {
                if (timeouts++ > 10) {
                    break;
                }
                continue;
            }
            if (timeouts) {
                timeouts--;
            }

            /* Send audio data to the audio codec. */
            bp = tcpbuf;
            rbytes = got;
            while (rbytes) {
                sent = _write(rip->ri_decoder, bp, rbytes);
                if (sent < 0) {
                    rip->ri_status |= RSTAT_STOP;
                    LogMsg(LOG_WARN, "Codec write failed\n");
                    break;
                }
                rbytes -= sent;
                bp += sent;
            }

            /* Receive optional meta data. */
            if (sci->sci_metaint) {
                sci->sci_metapos -= got;
                if (sci->sci_metapos == 0) {
                    if (ProcessMetaData(sip->si_sock, sci, &rip->ri_status)) {
                        rip->ri_status |= RSTAT_STOP;
                        LogMsg(LOG_WARN, "Metadata sync lost\n");
                        break;
                    }
                    sci->sci_metapos = sci->sci_metaint;
                }
            }

            /* Force decoder start after 10 seconds. */
            if (buffering) {
                int pbstat;
                _ioctl(rip->ri_decoder, AUDIO_GET_STATUS, &pbstat);
                if (pbstat) {
                    rip->ri_status |= RSTAT_RUNNING;
                    LogMsg(LOG_SHOUTCAST, "Playback started\n");
                    UserIfShowStatus(DIST_FORCE);
                    buffering = 0;
                }
            }
            if (buffering) {
                if(time(NULL) - buffering > MAX_WAIT_MP3BUF_FILLED) {
                    rip->ri_status &= ~RSTAT_BUFFERING;
                    _ioctl(rip->ri_decoder, AUDIO_PLAY, NULL);
                    NutSleep(500);
                }
            }
        }
        free(tcpbuf);

        /* Discard any buffered audio data. */
        _ioctl(rip->ri_decoder, AUDIO_CANCEL, NULL);
        rip->ri_status &= ~(RSTAT_BUFFERING | RSTAT_RUNNING);
        _close(rip->ri_decoder);
    }
}

/*!
 * \brief Create a SHOUTcast receiver instance.
 *
 * \param rip Pointer to the receiver information structure.
 *
 * \return 0 on success, -1 otherwise.
 */
int ShoutCastCreate(RECEIVERINFO * rip)
{
    /* Allocate local info structure. */
    if ((rip->ri_bcast = malloc(sizeof(SHOUTCASTINFO))) != NULL) {
        memset(rip->ri_bcast, 0, sizeof(SHOUTCASTINFO));

        /* Start the receiver thread. */
        if (NutThreadCreate("scast", ShoutCastThread, rip, SHOUTCAST_THREAD_STACK)) {
            /* Success! */
            return 0;
        }
        free(rip->ri_bcast);
        rip->ri_bcast = NULL;
    }
    /* Not enough memory. */
    return -1;
}

/*!
 * \brief Setup SHOUTcast receiver.
 *
 * Parses the header lines of the HTTP response.
 *
 * \return 0 on success, or -1 if the server responded with an error code
 *         or doesn't seem to provide a SHOUTcast stream.
 */
int ShoutCastSetup(RECEIVERINFO * rip, STATIONINFO *sip)
{
    SHOUTCASTINFO *sci = (SHOUTCASTINFO *) rip->ri_bcast;
    int i;
    char *cp;
    int rcode;

    /* Check if we really have a SHOUTcast server. */
    if (strlen(sip->si_header[0]) > 6 && strncmp(sip->si_header[0], "ICY", 3) == 0) {
        rcode = atoi(sip->si_header[0] + 4);
        sip->si_protocol = PROTOCOL_TYPE_SHOUTCAST;
    }
    /* Alternatively check for Icecast. */
    else if (strlen(sip->si_header[0]) > 11 && strncmp(sip->si_header[0], "HTTP/1", 6) == 0) {
        rcode = atoi(sip->si_header[0] + 9);
        sip->si_protocol = PROTOCOL_TYPE_ICECAST;
    }
    /* Reject unknown server responses. */
    else {
        return -1;
    }

    if (rcode != 200) {
        return -1;
    }
    sip->si_content = CONTENT_TYPE_MP3;
    sci->sci_metaint = 0;

    for (i = 1; sip->si_header[i]; i++) {
        if ((cp = strchr(sip->si_header[i], ':')) != NULL) {
            cp++;
            while (*cp == ' ') {
                cp++;
            }
            if (strncmp(sip->si_header[i], "icy-name:", 9) == 0) {
                /* Set the station name. */
                sip->si_name = cp;
            } else if (strncmp(sip->si_header[i], "icy-genre:", 10) == 0) {
                /* Set the station's genre. */
                sip->si_genre = cp;
            } else if (strncmp(sip->si_header[i], "icy-metaint:", 12) == 0) {
                /* Set the metadata interval. */
                sci->sci_metaint = atol(cp);
            } else if (strncmp(sip->si_header[i], "icy-br:", 7) == 0) {
                /* Set the bit rate. */
                sip->si_bitrate = atoi(cp);
            } else if (strncmp(sip->si_header[i], "content-type:", 13) == 0) {
                /* Check content type. Modify this to add more decoders. */
                if (strcmp(cp, "audio/mpeg") == 0) {
                    sip->si_content = CONTENT_TYPE_MP3;
                }
                else if (strcmp(cp, "application/ogg") == 0) {
                    sip->si_content = CONTENT_TYPE_OGG;
                }
                else if (strcmp(cp, "audio/aacp") == 0) {
                    sip->si_content = CONTENT_TYPE_AACP;
                }
                else {
                    sip->si_content = CONTENT_TYPE_UNKNOWN;
                }
            }
        }
    }
    return 0;
}

/*!
 * \brief Receiver plug-in reference structure.
 *
 * Used by the application to create a SHOUTcast receiver instance.
 */
RECEIVERPLUGIN rpiShoutcast = {
    ShoutCastCreate,    /*!< Plugin method rp_create. */
    ShoutCastSetup      /*!< Plugin method rp_setup. */
};

