/*
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file station.c
 * \brief Station.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <cfg/os.h>
#include <cfg/clock.h>
#include <dev/board.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>

#include <sys/version.h>
#include <sys/confnet.h>
#include <sys/atom.h>
#include <sys/heap.h>
#include <sys/thread.h>
#include <sys/timer.h>
#include <sys/event.h>
#include <sys/socket.h>

#if defined(USE_SOFTWARE_CODEC)
#include <hxmp3/mp3dec.h>
#endif

#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <pro/dhcp.h>

#include "config.h"
#include "logmsg.h"
#include "utils.h"
#include "shoutcast.h"

/*!
 * \brief Disconnect from a radio station.
 *
 * \param sip Pointer to a station information structure.
 */
void StationDisconnect(STATIONINFO *sip)
{
    if (sip) {
        LogMsg(LOG_STATION, "Disconnecting %.8s\n", sip->si_name);
        if (sip->si_sock) {
            NutTcpCloseSocket(sip->si_sock);
        }
        TcpReleaseHeaderLines(sip->si_header);
        free(sip);
    }
}

/*!
 * \brief Connect to a radio station.
 *
 * \param scp  Pointer to the station configuration table entry.
 * \param sidx Stream index.
 *
 * \return Pointer to a new station information structure. In
 *         case of an error, a NULL pointer is returned.
 */
STATIONINFO *StationConnect(RADIOSTATION * scp, int sidx)
{
    STATIONINFO *sip = NULL;
    TCPSOCKET *sock = NULL;
    u_short mss = MAX_TCPSEG_SIZE;
    u_short tcpbufsiz = MAX_TCPBUF_SIZE;
    u_long rx_to = MAX_TCPRCV_WAIT;
    int err = -1;
    int cr;
    char *line;
    HTTP_SCHEME *schm;

    if ((schm = HttpSchemeParse(scp->rs_uri[sidx])) == NULL) {
        return NULL;
    }

    /* Create a TCP socket. */
    if ((sock = NutTcpCreateSocket()) != NULL) {
        /* Set socket options. Failures are ignored. */
        NutTcpSetSockOpt(sock, TCP_MAXSEG, &mss, sizeof(mss));
        NutTcpSetSockOpt(sock, SO_RCVBUF, &tcpbufsiz, sizeof(tcpbufsiz));
        NutTcpSetSockOpt(sock, SO_RCVTIMEO, &rx_to, sizeof(rx_to));

        /* Connect the stream server. */
        LogMsg(LOG_STATION, "Trying entry %d, %s\n", sidx, scp->rs_uri[sidx]);
        cr = TcpHostConnect(sock, schm->schm_host, schm->schm_portnum);
        if (cr == 0) {
            LogMsg(LOG_STATION, "Connected %s:%u\n", schm->schm_host, schm->schm_portnum);

            /* Create a new station info structure. */
            if ((sip = malloc(sizeof(STATIONINFO))) != NULL) {
                memset(sip, 0, sizeof(STATIONINFO));
                sip->si_sock = sock;
                sip->si_scp = scp;

                /*
                 * Send the HTTP request.
                 */
                line = malloc(256);

                if (proxy.proxy_port) {
                    /* A proxy requires an absolute URI. */
                    sprintf(line, "GET http://%s/", scp->rs_uri[sidx]);
                }
                else {
                    strcpy(line, "GET /");
                }
                if (schm->schm_path) {
                    /* URL contains a specific path. */
                    strcat(line, schm->schm_path);
                }
                strcat(line, " HTTP/1.0\r\n");
                err = TcpPutString(sock, line);

                if (err == 0) {
                    /* HTTP 1.1 requires this. So just in case... */
                    sprintf(line, "Host: %s\r\n", schm->schm_host);
                    err = TcpPutString(sock, line);
                }

                if (err == 0) {
                    /* Some SHOUTcast servers seem to require a user-agent line. 
                       "Eat chalk" to get in. */
                    TcpPutString(sock, "User-Agent: WinampMPEG/2.7\r\n" /* */
                        "Accept: */*\r\n"  /* */
                        "Icy-MetaData: 1\r\n"  /* */
                        "Connection: close\r\n\r\n");
                }

                free(line);

                /* Read the servers response, collecting all header lines. */
                if (TcpGetHeaderLines(sock, &sip->si_header) <= 0) {
                    err = -1;
                }
            }
        }

        /* Release resources in case of any error. */
        if (err) {
            if (sip) {
                StationDisconnect(sip);
                sip = NULL;
            }
            else {
                NutTcpCloseSocket(sock);
            }
        }
    }
    HttpSchemeRelease(schm);

    return sip;
}
