#ifndef _STATION_H_
#define _STATION_H_

/*
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file station.h
 * \brief Radio station connect and disconnect routines.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <sys/socket.h>
#include "config.h"

#ifndef MAXNUM_STREAMS
#if defined (AT91SAM9260_EK) || defined(ELEKTOR_IR1)
#define MAXNUM_STREAMS      32
#else
#define MAXNUM_STREAMS      4
#endif
#endif

/*! \name Server Protocol Types */
/*@{*/
/*! \brief Protocol not determined. */
#define PROTOCOL_TYPE_UNKNOWN       0
/*! \brief Receiver is connected to a SHOUTcast server. */
#define PROTOCOL_TYPE_SHOUTCAST     1
/*! \brief Receiver is connected to an Icecast server. */
#define PROTOCOL_TYPE_ICECAST       2
/*@}*/

/*! \name Server Content Types */
/*@{*/
/*! \brief Unknown content. */
#define CONTENT_TYPE_UNKNOWN        0
/*! \brief MP3 encoded audio data. */
#define CONTENT_TYPE_MP3            1
/*! \brief Ogg Vorbis encoded audio data. */
#define CONTENT_TYPE_OGG            2
/*! \brief AAC+ encoded audio data. */
#define CONTENT_TYPE_AACP           3
/*@}*/

/*!
 * \brief Station configuration structure.
 *
 * This structure is passed to StationConnect().
 */
typedef struct {
    /*! \brief Symbolic station name.
     */
    char *rs_name;

    /*! \brief Number of available streams.
     */
    int rs_streams;

    /*! \brief Uniform resource locators of the streams.
     */
    char * rs_uri[MAXNUM_STREAMS];
} RADIOSTATION;

/*!
 * \brief Station information structure.
 *
 * An instance of this structure is created when calling StationConnect().
 */
typedef struct _STATIONINFO {
    /*! \brief TCP socket of this stream. */
    TCPSOCKET *si_sock;
    /*! \brief Station type, PROTOCOL_TYPE_. */
    u_int si_protocol;
    /*! \brief Player type, CONTENT_TYPE_. */
    u_int si_content;
    /*! \brief The station name. */
    char *si_name;
    /*! \brief The station's genre. */
    char *si_genre;
    /*! \brief Stream bit rate. */
    u_int si_bitrate;
    /*! \brief Header lines, collected during connect. */
    char **si_header;
    /*! \brief Pointer to the station configuration table entry. */
    RADIOSTATION * si_scp;
} STATIONINFO;

__BEGIN_DECLS
/* Prototypes */

extern STATIONINFO *StationConnect(RADIOSTATION * radio, int sidx);
extern void StationDisconnect(STATIONINFO *sip);

__END_DECLS
/* End of prototypes */
#endif
