/*
 * Copyright (C) 2006-2007 by egnite Software GmbH. All rights reserved.
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 */

/*!
 * \file utils.c
 * \brief Utility routines.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include "config.h"

#include <sys/heap.h>
#include <sys/timer.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <arpa/inet.h>
#include <netdb.h>

#include "logmsg.h"
#include "utils.h"

#ifndef MAX_HEADERLINE_SIZE
#define MAX_HEADERLINE_SIZE     255
#endif

/*!
 * \brief Connect TCP host.
 */
int TcpHostConnect(TCPSOCKET * sock, CONST char * rhost, u_short port)
{
    u_long rip;

    if (proxy.proxy_port) {
        port = proxy.proxy_port;
        rhost = proxy.proxy_host;
    }
    /* Get remote host IP. */
    rip = inet_addr(rhost);
    if (rip == (u_long)-1 || rip == 0) {
        rip = NutDnsGetHostByName((u_char *)rhost);
        if (rip == 0) {
            LogMsg(LOG_WARN, "DNS query for %s failed\n", rhost);
            return -1;
        }
    }
    if (NutTcpConnect(sock, rip, port)) {
        LogMsg(LOG_WARN, "Connection failed with error %d\n", NutTcpError(sock));
        return -1;
    }
    return 0;
}

FILE *TcpStreamAccept(TCPSOCKET * sock, u_short port, char *mode)
{
    FILE *stream = NULL;
    u_long avail;
    int wcntr;

    /* Wait for connecting client. */
    if (NutTcpAccept(sock, port)) {
        LogMsg(LOG_WARN, "Failed to listen at TCP port %u\n", port);
        return NULL;
    }

    /* Wait for required resources. */
    for (wcntr = 0; wcntr < 10; wcntr++) {
        avail = NutHeapAvailable();
        if (avail >= 8192) {
            stream = _fdopen((int)((uptr_t) sock), mode);
            if (stream) {
                break;
            }
            LogMsg(LOG_WARN, "Failed to create stream\n");
        }
        else {
            LogMsg(LOG_WARN, "Only %lu bytes free\n", avail);
        }
        NutSleep(500);
    }
    return stream;
}

/*!
 * \brief Get a specified number of characters from TCP socket connection.
 *
 * The low level NutTcpReceive() routine may return less than the number
 * of bytes specified. This routine makes sure, that the complete buffer
 * is filled. However, an additional pointer to a status is passed to
 * this routine. If the status changes, the routine will be aborted.
 *
 * Any TCP receive timeout is ignored, but required in order to detect
 * status changes.
 *
 * \param sock   Socket descriptor. This pointer must have been retrieved 
 *               by calling NutTcpCreateSocket(). In addition a connection 
 *               must have been established by calling NutTcpConnect() or 
 *               NutTcpAccept().
 * \param buff   Pointer to the buffer.
 * \param size   Size of the buffer.
 * \param status Pointer to a status.
 *
 * \return 0 if the buffer is filled. Otherwise -1 is returned.
 */
int TcpGetBuffer(TCPSOCKET * sock, char * buff, u_int size, u_int *status)
{
    u_int initial = *status;
    int got;

    while (size) {
        if (*status != initial || (got = NutTcpReceive(sock, buff, size)) < 0) {
            /* Status change or receive error. */
            return -1;
        }
        size -= got;
        buff += got;
    }
    return 0;
}

int TcpGetTag(TCPSOCKET * sock, char * tag, int size)
{
    int rc = 0;
    int got;
    char *cp = tag;
    int in_tag = 0;
    int in_spc = 0;
    int in_qte = 0;

    if (size > 0) {
        for (;;) {
            if ((got = NutTcpReceive(sock, cp, 1)) <= 0) {
                rc = -1;
                break;
            }
            if (!in_tag) {
                if (!in_qte) {
                    if (*cp == '"') {
                        in_qte = 1;
                    }
                    else if (*cp == '<') {
                        in_tag = 1;
                    }
                }
            }
            else if (in_qte) {
            }
            else if (isspace(*cp)) {
                if (!in_spc) {
                    *cp++ = ' ';
                    rc++;
                    in_spc = 1;
                }
            }
            else {
                in_spc = 0;
                if (*cp == '>') {
                    *cp = 0;
                    break;
                }
                if (rc < size) {
                    rc++;
                    cp++;
                }
            }
        }
    }
    return rc;
}

char * LocateAttributeString(CONST char * tag, CONST char * name)
{
    CONST char *cp;
    size_t nlen = strlen(name);
    int in_qte = 0;

    cp = tag;
    while (*cp) {
        /* Skip leading spaces. */
        if (isspace(*cp)) {
            cp++;
            continue;
        }
        if (strncmp(cp, name, nlen) == 0 && *(cp + nlen) == '=') {
            return (char *)(cp + nlen + 2);
        }
        while (*cp) {
            if (!in_qte) {
                if (isspace(*cp)) {
                    break;
                }
                if (*cp == '"') {
                    in_qte = 1;
                }
            }
            else {
                if (*cp == '"') {
                    in_qte = 0;
                }
            }
            cp++;
        }
    }
    return NULL;
}

/*!
 * \brief Get line from TCP socket connection.
 *
 * ASCII codes below 32 or above 127 are discarded.
 *
 * \param sock Socket descriptor. This pointer must have been retrieved 
 *             by calling NutTcpCreateSocket(). In addition a connection 
 *             must have been established by calling NutTcpConnect() or 
 *             NutTcpAccept().
 * \param line Pointer to the buffer that receives the string.
 * \param size Size of the buffer. If the received line doesn't fit,
 *             then remaining characters will be discarded.
 *
 * \return Number of characters stored in the line buffer or -1 in case
 *         of an error or timeout.
 */
int TcpGetLine(TCPSOCKET * sock, char * line, u_short size)
{
    int rc = 0;
    int got;
    char *cp = line;

    if (size > 0) {
        for (;;) {
            if ((got = NutTcpReceive(sock, cp, 1)) <= 0) {
                rc = -1;
                break;
            }
            if (*cp == '\n') {
                *cp = 0;
                break;
            }
            if (*cp >= ' ' && rc < (int) size) {
                rc++;
                cp++;
            }
        }
    }
    return rc;
}

/*!
 * \brief Send complete string via TCP.
 *
 * The low level NutTcpSend() routine limits the number of bytes to
 * the maximum segment size of the connection. This routine makes
 * sure, that the complete string is transmitted.
 *
 * \param sock Socket descriptor. This pointer must have been retrieved 
 *             by calling NutTcpCreateSocket(). In addition a connection 
 *             must have been established by calling NutTcpConnect() or 
 *             NutTcpAccept().
 * \param str  Pointer to a buffer containing the string to send.
 *
 * \return 0 on success. Otherwise -1 is returned.
 */
int TcpPutString(TCPSOCKET * sock, char * str)
{
    int len = (int)strlen(str);
    int c;

    while(len) {
        if ((c = NutTcpSend(sock, str, (u_short)len)) <= 0) {
            break;
        }
        len -= c;
        str += c;
    }
    return len ? -1 : 0;
}

/*!
 * \brief Get header line array from TCP socket connection.
 *
 * Receives lines from a TCP stream until the first empty line appears
 * and stores all lines in an array of strings. The empty line is not
 * stored, but the last pointer of the array is set to NULL.
 *
 * The routine allocates the array and all string. The caller should use
 * TcpReleaseHeaderLines() to release all allocated memory.
 *
 * \param sock  Socket descriptor. This pointer must have been retrieved 
 *              by calling NutTcpCreateSocket(). In addition a connection 
 *              must have been established by calling NutTcpConnect() or 
 *              NutTcpAccept().
 * \param array Pointer to a string array pointer, which receives the
 *              address of the newly allocated string array. If no
 *              non-empty lines were received or if an error occured,
 *              then the string array pointer will be set to NULL.
 *
 * \return The number of header lines stored. In case of an error, the
 *         function result is -1 instead.
 */
int TcpGetHeaderLines(TCPSOCKET * sock, char ***array)
{
    int rc = -1;
    struct LILI_ {
        struct LILI_ *ll_next;
        char *ll_line;
    };
    struct LILI_ *root = NULL;
    struct LILI_ *link = NULL;
    struct LILI_ **next = &root;
    char *buf;
    int len;

    /*
     * Allocate a line buffer. On success, build a linked list of
     * incoming lines. Stop on errors or at the first empty line.
     */
    if ((buf = malloc(MAX_HEADERLINE_SIZE)) != NULL) {
        for (;;) {
            if ((len = TcpGetLine(sock, buf, MAX_HEADERLINE_SIZE)) == 0) {
                /* Empty line. */
                break;
            }
            if (len < 0) {
                /* Error occured. */
                rc = -1;
                break;
            }
            /* Allocate a linked list item. */
            if ((*next = malloc(sizeof(struct LILI_))) == NULL) {
                break;
            }
            /* Initially set the next link to NULL. */
            (*next)->ll_next = NULL;
            /* Copy the line to the linked list item. */
            (*next)->ll_line = malloc(len + 1);
            memcpy((*next)->ll_line, buf, len);
            (*next)->ll_line[len] = '\0';
            /* Set pointer to the next link. */
            next = &((*next)->ll_next);
            rc++;
        }
        free(buf);
    }

    if (rc > 0) {
        /* Allocate a new string array. */
        *array = malloc((rc + 1) * sizeof(char *));
        rc = 0;
    }
    else {
        /* Error or single empty line only. Set array pointer to NULL. */
        *array = NULL;
    }

    /*
     * This loop serves two purposes. It moves the strings from the
     * linked list to the string array (if one had been allocated)
     * and it releases all memory allocated for the linked list.
     */
    while (root) {
        if (*array) {
            /* Move string to the array. */
            (*array)[rc] = root->ll_line;
            rc++;
        }
        else {
            /* No array. Just release the string. */
            free(root->ll_line);
        }
        /* Get the next link and release the current entry. */
        link = root;
        root = root->ll_next;
        free(link);
    }

    /* NULL marks the end of the array. */
    if (*array) {
        (*array)[rc] = NULL;
    }
    return rc;
}

/*!
 * \brief Release previously allocated memory for the header lines.
 *
 * Should be called to release all memory allocated by TcpGetHeaderLines().
 *
 * \param array String array to be released.
 */
void TcpReleaseHeaderLines(char **array)
{
    char **ap = array;

    if (ap) {
        /* Thanks, Dany. */
        while (*ap) {
            free(*ap);
            ap++;
        }
        free(array);
    }
}

void HttpSchemeRelease(HTTP_SCHEME *schm)
{
    if (schm) {
        if (schm->schm_uri) {
            free(schm->schm_uri);
        }
        free(schm);
    }
}

/* [<user>[:<password>]@]<host>[:<port>][/<path>] */
HTTP_SCHEME *HttpSchemeParse(CONST char *uri)
{
    HTTP_SCHEME *schm = NULL;
    char *cp;

    /* Create a blank scheme structure. */
    if (*uri && (schm = malloc(sizeof(HTTP_SCHEME))) != NULL) {
        memset(schm, 0, sizeof(HTTP_SCHEME));

        /* Create a local copy of the URI string. */
        if ((schm->schm_uri = strdup(uri)) != NULL) {
            /* Split the local copy. */
            schm->schm_host = schm->schm_uri;
            for (cp = schm->schm_uri; *cp; cp++) {
                if (*cp == ':') {
                    *cp = '\0';
                    schm->schm_port = cp + 1;
                }
                else if (*cp == '/') {
                    *cp = 0;
                    schm->schm_path = cp + 1;
                    break;
                }
                else if (*cp == '@') {
                    *cp = 0;
                    schm->schm_user = schm->schm_host;
                    schm->schm_pass = schm->schm_port;
                    schm->schm_host = cp + 1;
                    schm->schm_port = NULL;
                }
            }
            if (schm->schm_port) {
                schm->schm_portnum = (u_short)atoi(schm->schm_port);
            }
            else {
                schm->schm_portnum = 80;
            }
            return schm;
        }
    }
    HttpSchemeRelease(schm);
    return NULL;
}

/*!
 * \brief Switch LED0 on or off.
 *
 * On the SAM7X-EK the LED labeled DS4 is used.
 *
 * On the SAM9260-EK we use the only one user LED.
 *
 * \param on If 0, then the LED will be switched off. Otherwise it will be lit.
 */
void Led0(int on)
{
#ifdef AT91SAM7X_EK
    outr(PIOB_PER, _BV(22));
    outr(PIOB_OER, _BV(22));
    if (on) {
        outr(PIOB_CODR, _BV(22));
    }
    else {
        outr(PIOB_SODR, _BV(22));
    }
#elif AT91SAM9260_EK
    outr(PIOA_PER, _BV(6));
    outr(PIOA_OER, _BV(6));
    if (on) {
        outr(PIOA_CODR, _BV(6));
    }
    else {
        outr(PIOA_SODR, _BV(6));
    }
#endif
}

/*!
 * \brief Create a copy of a string.
 * 
 * Allocates sufficient memory from heap for a copy of the string
 * does the copy.
 *
 * Missing Nut/OS function.
 *
 * \param str Pointer to the string to copy.
 *
 * \return A pointer to the new string or NULL if allocating memory failed.
 */
char *strdup(CONST char *str)
{
    size_t siz;
    char *copy;

    siz = strlen(str) + 1;
    if ((copy = malloc(siz)) == NULL) {
        return NULL;
    }
    memcpy(copy, str, siz);

    return copy;
}
