/*
 * Copyright (C) 2008 by egnite GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 *
 */

/*!
 * \file xmlserv.c
 * \brief XML server interface.
 *
 * \verbatim
 *
 * $Log$
 *
 * \endverbatim
 */

#include <cfg/os.h>
#include <cfg/memory.h>

#include "config.h"

#ifdef USE_HTTPSERVER

#include <stdlib.h>
#include <string.h>
#include <io.h>

#include <netinet/tcp.h>

#include <dev/board.h>
#include <dev/vscodec.h>

#include <sys/heap.h>
#include <sys/event.h>
#include <sys/timer.h>

#include <pro/uxml.h>

#include "logmsg.h"
#include "utils.h"
#include "favlist.h"
#include "userif.h"
#include "xmlserv.h"

/* Service thread counter. */
static int xmld_tc;

/* Info event queue. */
static HANDLE xml_infoq;

static int StartReportServiceThread(void);

void XmlRefresh(void)
{
    NutEventBroadcast(&xml_infoq);
}

/*
 * Process XML commands.
 */
static int XmlProcessCommands(FILE *stream)
{
    UXML_NODE *cmd_tree = NULL;
    UXML_NODE *node;
    UXML_ATTRIB *attr;
    char *cmdp;
    char *valp;

    char *f_tags[] = { "cmd", NULL };
    char *f_attribs[] = { "name", "value", NULL };

    /* Create tree from incoming XML stream. */
    if ((cmd_tree = UxmlParseStream(stream, f_tags, f_attribs)) == NULL) {
        if (ferror(stream)) {
            /* Probably disconnected. */
            return -1;
        }
        /* Probably a timeout. */
        LogMsg(LOG_XMLCD, "Connection timed out\n");
        return 0;
    }

    for (node = cmd_tree; node; node = node->xmln_next) {
        attr = node->xmln_attribs;
        cmdp = NULL;
        valp = NULL;
        while (attr) {
            if (strcasecmp(attr->xmla_name, "name") == 0) {
                cmdp = attr->xmla_value;
            }
            else if (strcasecmp(attr->xmla_name, "value") == 0) {
                valp = attr->xmla_value;
            }
            attr = attr->xmla_next;
        }
        if (cmdp) {
            LogMsg(LOG_XMLCD, "XML cmd '%s%s%s'\n", cmdp, valp ? "=" : "", valp ? valp : "");
            if (strcasecmp(cmdp, "gain") == 0) {
                if (valp) {
                    webradio.wr_gain = atoi(valp);
                    webradio.wr_cfgupd = 10;
                    if (webradio.wr_rip == NULL || /* */
                        webradio.wr_rip->ri_decoder == -1 || /* */
                        _ioctl(webradio.wr_rip->ri_decoder, AUDIO_SET_PLAYGAIN, &webradio.wr_gain) != 0) {
                            LogMsg(LOG_ERROR, "No volume control\n");
                    }
                    XmlRefresh();
                }
            }
        }
    }
    /* Release the XML tree and return. */
    UxmlTreeDestroy(cmd_tree);
    return 0;
}

/*
 * Send XML report.
 */
static int XmlProcessReports(FILE *stream)
{
    char *title = malloc(256);

    fprintf(stream, "<radioinfo>");
    if (title) {
        fprintf(stream, "<title value=\"%s\"></title>", UserIfGetDisplayText(title, 256));
    }
    fprintf(stream, "<gain value=\"%d\" min=\"%d\" max=\"%d\"></gain>", webradio.wr_gain, AUDIO_DAC_MIN_GAIN, AUDIO_DAC_MAX_GAIN);
    fprintf(stream, "</radioinfo>");
    fputc(0, stream);
    return fflush(stream);
}

/*! \fn XmlCmdService(void *arg)
 * \brief XML command service thread.
 *
 * This thread processes XML command request.
 */
THREAD(XmlCmdService, arg)
{
    TCPSOCKET *sock;
    FILE *stream;

    /*
     * Each loop serves a single connection.
     */
    LogMsg(LOG_XMLCD, "Cmd daemon running\n");
    for (;;) {
        /* Create a socket. */
        if ((sock = NutTcpCreateSocket()) == 0) {
            LogMsg(LOG_ERROR, "No socket\n");
            NutSleep(5000);
            continue;
        }

        /* Set socket options. Silently ignore any error. */
#ifdef XMLD_MAX_SEGSIZE
        {
            u_short mss = XMLD_MAX_SEGSIZE;
            NutTcpSetSockOpt(sock, TCP_MAXSEG, &mss, sizeof(mss));
        }
#endif
#ifdef XMLD_TCP_BUFSIZE
        {
            u_short tcpbufsiz = XMLD_TCP_BUFSIZE;
            NutTcpSetSockOpt(sock, SO_RCVBUF, &tcpbufsiz, sizeof(tcpbufsiz));
        }
#endif
#ifdef XMLD_TCP_TIMEOUT
        {
            u_long tmo = XMLD_TCP_TIMEOUT;
            NutTcpSetSockOpt(sock, SO_RCVTIMEO, &tmo, sizeof(tmo));
        }
#endif

        LogMsg(LOG_XMLCD, "Waiting for cmd client\n");
        stream = TcpStreamAccept(sock, XMLD_CMD_TCP_PORT, "r+b");
        if (stream) {
            LogMsg(LOG_XMLCD, "Connected cmd client\n");
            while (XmlProcessCommands(stream) == 0) {
            }
            LogMsg(LOG_HTTPD, "Disconnecting cmd client\n");
            fclose(stream);
        } 
        NutTcpCloseSocket(sock);
    }
}


/*! \fn XmlRepoService(void *arg)
 * \brief XML report thread.
 *
 * This thread sends XML reports to a connected client.
 */
THREAD(XmlRepoService, arg)
{
    TCPSOCKET *sock;
    FILE *stream;
    int refresh;

    LogMsg(LOG_XMLRD, "Repo daemon started\n");
    for (;;) {
        /* Create a socket. */
        if ((sock = NutTcpCreateSocket()) == 0) {
            LogMsg(LOG_WARN, "No sockets\n");
            NutSleep(1000);
            continue;
        }

        /* Set socket options. Silently ignore any error. */
#ifdef XMLD_MAX_SEGSIZE
        {
            u_short mss = XMLD_MAX_SEGSIZE;
            NutTcpSetSockOpt(sock, TCP_MAXSEG, &mss, sizeof(mss));
        }
#endif
#ifdef XMLD_TCP_BUFSIZE
        {
            u_short tcpbufsiz = XMLD_TCP_BUFSIZE;
            NutTcpSetSockOpt(sock, SO_RCVBUF, &tcpbufsiz, sizeof(tcpbufsiz));
        }
#endif
#ifdef XMLD_TCP_TIMEOUT
        {
            u_long tmo = XMLD_TCP_TIMEOUT;
            NutTcpSetSockOpt(sock, SO_RCVTIMEO, &tmo, sizeof(tmo));
        }
#endif

        LogMsg(LOG_XMLRD, "Waiting for repo client\n");
        stream = TcpStreamAccept(sock, XMLD_REPO_TCP_PORT, "r+b");
        if (stream) {
            LogMsg(LOG_XMLRD, "Connected repo client\n");
#ifdef USE_DYNAMIC_THREADS
            /* Start a new thread and let it run. */
            StartReportServiceThread();
            NutSleep(1);
#endif
            refresh = 30;
            for (;;) {
                /* Use timeout probing to detect broken connections. */
                if (++refresh < 30 && NutEventWait(&xml_infoq, 1000)) {
                    fputc(0, stream);
                    if (fflush(stream)) {
                        break;
                    }
                }
                else {
                    refresh = 0;
                    if (XmlProcessReports(stream)) {
                        break;
                    }
                }
            }
            LogMsg(LOG_XMLRD, "Disconnecting repo client\n");
            fclose(stream);
        }
        NutTcpCloseSocket(sock);

#ifdef USE_DYNAMIC_THREADS
        /* If enough threads are running, stop this one. */
        if (xmld_tc >= XMLD_MIN_THREADS) {
            xmld_tc--;
            LogMsg(LOG_XMLRD, "Stop repo daemon\n");
            NutThreadExit();
        }
#endif
    }
}

/*
 * Start a XML daemon thread.
 */
static int StartReportServiceThread(void)
{
#ifdef USE_DYNAMIC_THREADS
    if (xmld_tc >= XMLD_MAX_THREADS) {
        return 0;
    }
#endif

    if (NutThreadCreate("xmlrepo", XmlRepoService, NULL, XMLD_SERVICE_STACK) == NULL) {
        return -1;
    }
    xmld_tc++;

    return 0;
}

/*!
 * \brief XML Server Start.
 *
 * Starts a hard coded number of reporter threads and one command thread.
 *
 * \return 0 on success, -1 otherwise.
 */
int XmlServerStart(void)
{
    int i;

    /*
     * Start some report server threads.
     */
    LogMsg(LOG_XMLRD, "Starting %d XML repo daemons\n", XMLD_MIN_THREADS);
    for (i = 0; i < XMLD_MIN_THREADS; i++) {
        if (StartReportServiceThread()) {
            LogMsg(LOG_ERROR, "Thread start failed\n");
            return -1;
        }
    }

    /*
     * Start one command server thread.
     */
    LogMsg(LOG_XMLCD, "Starting XML cmd daemon\n");
    if (NutThreadCreate("xmlcmd", XmlCmdService, NULL, XMLD_SERVICE_STACK) == NULL) {
        LogMsg(LOG_ERROR, "Thread start failed\n");
        return -1;
    }
    xmld_tc++;
    NutSleep(1);

    return 0;
}

#endif /* USE_HTTPSERVER */
